AJS.test.require("com.atlassian.jira.jira-projects-plugin:pinnablenavigator-test", function() {
    "use strict";

    var jQuery = require("jquery");
    var wrmContextPath = require("wrm/context-path");
    var browser = require("jira/util/browser");
    var PinnedListView = require("jira/projects/components/pinnable-navigator/views/list-view/pinned-list-view");
    var PinnableNavFactory = require("jira/projects/components/pinnable-navigator/pinnable-navigator-factory");

    var ITEM_1 = {id: "item1", label: "Item 1", link: "http://www.atlassian.com"};
    var ITEM_2 = {id: "item2", label: "Item 2", link: "http://jira.atlassian.com"};
    var ITEM_3 = {id: "item3", label: "Item 3", link: "http://jira.atlassian.com"};
    var ITEMS = [
        {items: [ITEM_1]},
        {items: [ITEM_2], allowDelete: true}
    ];

    module("jira/projects/components/pinnable-navigator/pinnable-navigator-factory", {
        setup: function() {
            jQuery("<div id='reports-subnav-trigger'/>").appendTo('#qunit-fixture');
            jQuery("<div id='reports-pin-holder'/>").appendTo('#qunit-fixture');
            jQuery("<div id='reports-subnav-opts'/>").appendTo('#qunit-fixture');
            this.sandbox = sinon.sandbox.create();
            this.sandbox.stub(browser, "reloadViaWindowLocation");
            this.server = sinon.fakeServer.create();
        },

        teardown: function() {
            this.sandbox.restore();
            this.server.restore();
        },

        controllerWithItems: function(items, options) {
            options = options || {};
            return PinnableNavFactory.create({
                id: "reports",
                itemGroups: items,
                selectedItem: "item1",
                isPinned: options.isPinned,
                pinnedView: PinnedListView,
                triggerHolderId: "reports-subnav-trigger",
                pinnedHolderId: "reports-pin-holder",
                unpinnedHolderId: "reports-subnav-opts",
                title: "Reports",
                manageText: "Manage Reports"
            });
        },

        selectItem: function(item) {
            this.contentPlaceholder.find("a[data-item-id='" + item.id + "']").click();
            this.contentPlaceholder.find(".subnav-opts").trigger("aui-dropdown2-hide");
        },

        assert: {
            isPinned: function (controller) {

                // All the unpinned views should be clean up
                ok(!controller.options.triggerView.$el.text().length,
                    "Expected the trigger view element to contain text");

                equal(controller.options.triggerView.$el.css("display"), "none",
                    "Expected the trigger view element to be hidden");

                ok(!controller.options.unpinnedView.$el.children().length,
                    "Expected the unpinned view (dropdown) to be emptied");


                // Pinned views populated
                ok(!!controller.options.pinnedView.$el.closest("body").length,
                    "Expected pinned view to be attached to dom");
                ok(!!controller.options.pinnedView.$el.children().length,
                    "Expected the pinned view to be populated with context");
            },
            isUnpinned: function (controller) {

                // Pinned views should be cleaned up
                ok(!controller.options.pinnedView.$el.closest("body").length,
                    "Expected pinned view to be detached from dom");

                // All unpinned views should be populated
                ok(!!controller.options.triggerView.$el.text().length,
                    "Expected trigger view element to contain text");

                ok(!!controller.options.unpinnedView.$el.children().length,
                    "Expected the unpinned view (dropdown) to be populated");
            }
        }
    });

    test("Init state of pinned is respected", function () {
        var controller = this.controllerWithItems([{items: [ITEM_1]}], {isPinned: true});
        controller.show();
        this.assert.isPinned(controller);
    });

    test("Init state of un-pinned is respected", function () {
        var controller = this.controllerWithItems([{items: [ITEM_1]}], {isPinned: false});
        controller.show();
        this.assert.isUnpinned(controller);
    });


    test("Renders the trigger view as a button", function() {
        var controller = this.controllerWithItems(ITEMS);

        controller.show();

        ok(controller.options.triggerView.$el.html().length > 0);
    });

    test("The navigator will be pinned when 'Keep menu open' is clicked", function() {
        var controller = this.controllerWithItems(ITEMS, {isPinned: false});
        controller.show();
        controller.options.unpinnedView.$el.find('.js-pin-toggler').click();
        this.assert.isPinned(controller);
    });

    test("The navigator will be unpinned when 'X' icon is clicked", function() {
        var controller = this.controllerWithItems(ITEMS, {isPinned: true});
        controller.show();
        controller.options.pinnedView.$el.find('.js-unpin-button').click();
        this.assert.isUnpinned(controller);
    });


    test("Saving the pinned state", function() {
        var controller = this.controllerWithItems(ITEMS);

        controller.show();

        sinon.spy(controller, "saveState");
        controller.options.unpinnedView.trigger('navigatorPinned');

        sinon.assert.calledOnce(controller.saveState);

        equal(this.server.requests[1].url, wrmContextPath() + "/rest/projects/1.0/subnav/reports/pin");
    });

    test("Saving the unpinned state", function() {
        var controller = this.controllerWithItems(ITEMS);

        controller.show();

        sinon.spy(controller, "saveState");
        controller.options.pinnedView.trigger('navigatorUnPinned');

        sinon.assert.calledOnce(controller.saveState);

        equal(this.server.requests[1].url, wrmContextPath() + "/rest/projects/1.0/subnav/reports/unpin");
    });

    test("Clicking on manage button in unpinned view will fire 'manageDialogOpened' event", function() {
        var onClick = this.spy();
        var controller = this.controllerWithItems(ITEMS);
        controller.options.unpinnedView.listenTo(controller.options.unpinnedView, "manageDialogOpened", onClick);
        controller.options.unpinnedView.render();
        controller.options.unpinnedView.putListintoView();
        controller.options.unpinnedView.$el.find(".js-manage-button").click();
        sinon.assert.calledOnce(onClick);

        jQuery('.aui-iconfont-close-dialog').click();
    });

    test("Updating the views with given data", function() {

        var controller = this.controllerWithItems(ITEMS);
        controller.update([{items: [ITEM_3]}]);

        var itemGroups = controller.options.model.get('itemGroups');

        equal(itemGroups[0].items.length,1);
        equal(itemGroups[0].items[0].id,ITEM_3.id);
    });

    test("Updating the views with given data, and selected item", function() {
        var controller = this.controllerWithItems(ITEMS);
        controller.update([{items: [ITEM_3]}], "item3");
        var selectedItem = controller.options.model.get('selectedItem');
        equal(selectedItem.id, ITEM_3.id);
    });

    test("Set selected item", function() {
        var controller = this.controllerWithItems(ITEMS);
        controller.setSelectedItem(ITEM_2.id);

        var selectedItem = controller.options.model.get('selectedItem');

        equal(selectedItem.id,ITEM_2.id);
    });

    test("Hide method hides all views", function() {
        var controller = this.controllerWithItems(ITEMS);
        controller.options.pinnedView.$el.hide = sinon.stub();
        controller.options.triggerView.$el.hide = sinon.stub();
        controller.options.unpinnedView.$el.hide = sinon.stub();

        controller.hide();

        sinon.assert.calledOnce(controller.options.pinnedView.$el.hide);
        sinon.assert.calledOnce(controller.options.triggerView.$el.hide);
        sinon.assert.calledOnce(controller.options.unpinnedView.$el.hide);
    });
});
