define('jira/projects/sidebar/project-shortcuts/views/dialog-content', [
    "jira/util/formatter",
    "jira/flag",
    "jira/projects/sidebar/project-shortcuts/views/icon-picker-content",
    "jira/projects/sidebar/project-shortcuts/services/available-icons",
    "jira/projects/libs/marionette",
    "underscore"
], function(
    formatter,
    flag,
    IconPickerContentView,
    AvailableIcons,
    Marionette,
    _
) {
    "use strict";

    function formSubmit(e) {
        e.preventDefault();
        this.ensureProtocolPrefix();
        this.tryToAutomagicallyDeriveNameFromUrl();
        if (this.iconPickerContent) {
            this.iconPickerContent.onFormSubmit();
        }

        this.model.set("url", this.ui.url.val());
        this.model.set("name", this.ui.name.val());
        this.model.save();
    }

    var DialogContentView = Marionette.LayoutView.extend({
        ui: {
            form: "form",
            inputs: "input, button",
            submit: ".project-shortcuts-submit",
            cancel: ".project-shortcuts-cancel",
            url: ".project-shortcuts-url input",
            name: ".project-shortcuts-name input"
        },

        regions: {
            iconPicker: ".project-shortcuts-icon-picker-block"
        },

        events: {
            "click @ui.cancel": function cancel(e) {
                e.preventDefault();
                this.model.clear();
                this.setNameAutomagically = true;
                this.errorModel.clear();
                this.trigger("cancel");
            },
            "click @ui.submit": formSubmit,
            "submit @ui.form": formSubmit,
            "blur @ui.url": function ensureProtocol() {
                this.ensureProtocolPrefix();
                this.tryToAutomagicallyDeriveNameFromUrl();
            },
            "input @ui.url": function urlChange() {
                this.model.set("url", this.ui.url.val());
                this.tryToAutomagicallyDeriveNameFromUrl();
            },
            "input @ui.name": function nameChange() {
                this.setNameAutomagically = false;
                this.model.set("name", this.ui.name.val());
            },
            "keydown @ui.name": function nameKeydown(e) {
                if (!this.iconPickerContent) return;

                // tabindex doesn't work on Mac when Full Keyboard Access is disabled
                if (e.shiftKey && e.keyCode === AJS.keyCode.TAB) {
                    e.preventDefault();
                    this.iconPickerContent.focus();
                }
            },
            "keydown @ui.url": function nameKeydown(e) {
                if (!this.iconPickerContent) return;

                // tabindex doesn't work on Mac when Full Keyboard Access is disabled
                if (!e.shiftKey && e.keyCode === AJS.keyCode.TAB) {
                    e.preventDefault();
                    this.iconPickerContent.focus();
                }
            }
        },

        modelEvents: {
            "save:start": function saveStarted() {
                this.ui.inputs.prop("disabled", true);
                this.ui.submit.addClass("loading");
                this.ui.submit.spin({className: "spinner"});
            },
            "save:finish": function saveFinish() {
                this.ui.inputs.prop("disabled", false);
                this.ui.submit.removeClass("loading");
                this.ui.submit.spinStop({className: "spinner"});
            },
            "save:failure": function renderWithErrors() {
                this.render();
                if (this.errorModel.get("generalError")) {
                    flag({
                        type: "error",
                        title: formatter.I18n.getText('project.shortcuts.dialog.error.general.save'),
                        close: "auto",
                        body: this.errorModel.get("generalError")
                    });
                }
                if (this.errorModel.get("nameError")) {
                    this.ui.name.focus();
                }
                if (this.errorModel.get("urlError")) {
                    this.ui.url.focus();
                }

                return this;
            },
            "save:success": function saveSuccess() {
                this.setNameAutomagically = true;
            }
        },

        initialize: function initialize(options) {
            this.errorModel = options.errorModel;
            var currentName = this.model.get("name");
            this.setNameAutomagically = (currentName.length == 0);
            this.iconFactory = AvailableIcons;
        },

        serializeData: function serializeData() {
            return _.extend(this.model.toJSON(), {
                errors: this.errorModel.toJSON(),
                action: this.action,
                isWithIcon: this.iconFactory.isWithIcons()
            });
        },

        onRender: function () {
            if (this.iconFactory.isWithIcons()) {
                this.iconPickerContent = new IconPickerContentView({
                    model: this.model,
                    action: this.action,
                    observable: this
                });

                this.getRegion("iconPicker").show(this.iconPickerContent);
            } else {
                delete this.iconPickerContent;
            }
        },

        hideIconPicker: function() {
            if (!this.iconPickerContent) return;

            this.iconPickerContent.hideIconPicker();
        },

        setName: function setName(name) {
            this.ui.name.val(name);
            this.model.set("name", name);
        },
        setUrl: function setUrl(url) {
            this.ui.url.val(url);
            this.model.set("url", url);
        },
        ensureProtocolPrefix: function ensureProtocolPrefix() {
            var currentUrl = this.ui.url.val().trim();
            if (currentUrl.length > 0 && !DialogContentView.urlPattern.test(currentUrl)) {
                this.setUrl('http://' + currentUrl);
            }
        },
        tryToAutomagicallyDeriveNameFromUrl: function tryToAutomagicallyDeriveNameFromUrl() {
            var currentUrl = this.ui.url.val().trim();
            if (this.setNameAutomagically) {
                if (DialogContentView.urlOptionalProtocolPattern.test(currentUrl)) {
                    var matches = DialogContentView.urlOptionalProtocolPattern.exec(currentUrl);
                    this.setName(matches[3]);
                } else {
                    this.setName(currentUrl);
                }
            }
        }
    }, {
        urlPattern: /^[a-zA-Z0-9]+:(\/\/)?([^\/]*).*/,
        urlOptionalProtocolPattern: /^([a-zA-Z0-9]+:(\/\/)?)?([^\/]*).*/
    });

    return DialogContentView;
});

AJS.namespace('JIRA.Projects.Sidebar.ProjectShortcuts.Views.DialogContent', null, require('jira/projects/sidebar/project-shortcuts/views/dialog-content'));
