AJS.test.require( ["com.atlassian.jira.jira-projects-plugin:sidebar-expansion-manager", "com.atlassian.auiplugin:aui-sidebar"], function() {
    "use strict";

    var expansionManager = require('jira/projects/sidebar/expansion-manager');
    var LOCALSTORAGE_KEY = 'jira-sidebar-collapsed';
    var storage;
    var sidebar;
    var onChange;
    var beforeChange;
    var responsiveEvent;
    var userEvent;

    module("jira/projects/sidebar/expansion-manager", {
        setup: function() {
            this.context = AJS.test.mockableModuleContext();

            var ajsSidebar = AJS.sidebar();

            sinon.stub(ajsSidebar, 'collapse');
            sinon.stub(ajsSidebar, 'expand');
            sinon.stub(ajsSidebar, 'on');

            sidebar = {
                $el: [1],
                collapse: sinon.spy(function() {
                    sidebar.isCollapsed.returns(true);
                }),
                expand: sinon.spy(function() {
                    sidebar.isCollapsed.returns(false);
                }),
                on: sinon.spy(function(str, func) {
                    if (str.indexOf("-start") > 0) {
                        beforeChange = func
                    } else {
                        onChange = func;
                    }
                }),
                reflow: sinon.stub(ajsSidebar, 'reflow').returns(true),
                isViewportNarrow: sinon.stub(ajsSidebar, 'isViewportNarrow').returns(false),
                isCollapsed: sinon.stub(ajsSidebar, 'isCollapsed').returns(true)
            };

            responsiveEvent = {
                isResponsive: true,
                preventDefault: sinon.stub()
            };

            userEvent = {
                isResponsive: false,
                preventDefault: sinon.stub()
            };

            storage = {};
        }
    });

    test('Sidebar persist init', function () {
        expansionManager(sidebar, storage);

        ok(sidebar.on.calledWith('expand-end collapse-end'), 'Correct events must be bound');
    });

    test('Sidebar does not persist when viewport is narrow', function () {
        expansionManager(sidebar, storage);

        sidebar.isViewportNarrow.returns(true);
        onChange(userEvent);

        ok(!(LOCALSTORAGE_KEY in storage), 'Viewport is narrow, so nothing should be saved');
        sinon.assert.notCalled(sidebar.collapse);
        sinon.assert.calledOnce(sidebar.expand);
    });

    test('Sidebar does not expand when triggered by window resize and it was collapsed before', function () {
        expansionManager(sidebar, storage);

        sidebar.isViewportNarrow.returns(false);
        sidebar.collapse();
        onChange(userEvent);

        equal(storage[LOCALSTORAGE_KEY], 'true', 'Save sidebar collapsed');

        sidebar.isViewportNarrow.returns(true);
        onChange(responsiveEvent);

        equal(storage[LOCALSTORAGE_KEY], 'true', 'Save sidebar collapsed');

        sidebar.isViewportNarrow.returns(false);
        beforeChange(responsiveEvent);

        sinon.assert.calledOnce(responsiveEvent.preventDefault);
        equal(storage[LOCALSTORAGE_KEY], 'true', 'Save sidebar collapsed');
    });

    test('Sidebar expands when triggered by window resize and it was expanded before', function () {
        expansionManager(sidebar, storage);

        sidebar.isViewportNarrow.returns(false);
        sidebar.expand();
        onChange(userEvent);

        equal(storage[LOCALSTORAGE_KEY], 'false', 'Save sidebar expanded');

        sidebar.isViewportNarrow.returns(true);
        onChange(responsiveEvent);

        equal(storage[LOCALSTORAGE_KEY], 'false', 'Save sidebar expanded');

        sidebar.isViewportNarrow.returns(false);
        beforeChange(responsiveEvent);

        sinon.assert.notCalled(responsiveEvent.preventDefault);
        equal(storage[LOCALSTORAGE_KEY], 'false', 'Save sidebar expanded');
    });

    test('Sidebar persists correctly', function () {
        expansionManager(sidebar, storage);

        sidebar.isViewportNarrow.returns(false);
        onChange(userEvent);

        equal(storage[LOCALSTORAGE_KEY], 'false', 'Save sidebar expanded');

        sidebar.isCollapsed.returns(false);
        onChange(userEvent);
        equal(storage[LOCALSTORAGE_KEY], 'false', 'Save sidebar expanded');
    });

    test('Sidebar loads correctly from storage', function () {
        // Collapsed in storage
        sidebar.isViewportNarrow.returns(false);
        storage[LOCALSTORAGE_KEY] = 'true';

        expansionManager(sidebar, storage);

        ok(sidebar.collapse.calledOnce, 'Collapse sidebar');

        // Expanded in storage
        storage[LOCALSTORAGE_KEY] = 'false';

        expansionManager(sidebar, storage);
        ok(sidebar.expand.calledOnce, 'Expand sidebar');
    });

    test('Sidebar can be collapsed by default with data provider', function() {
        this.context.mock('wrm/data', {
            claim: sinon.stub().withArgs('sidebar-collapsed-by-default').returns(true)
        });
        expansionManager = this.context.require('jira/projects/sidebar/expansion-manager');
        expansionManager(sidebar, storage);

        ok(sidebar.collapse.calledOnce, 'Collapsed sidebar by default');
    });

    test('Sidebar doesnt collapse by default', function() {
        this.context.mock('wrm/data', {
            claim: sinon.stub().withArgs('sidebar-collapsed-by-default').returns(undefined)
        });
        expansionManager = this.context.require('jira/projects/sidebar/expansion-manager');
        expansionManager(sidebar, storage);

        ok(sidebar.expand.calledOnce, 'Collapsed sidebar by default');
        sinon.assert.notCalled(responsiveEvent.preventDefault);
    });

});
