AJS.test.require(["com.atlassian.jira.jira-projects-plugin:sidebar-component"], function(){
    "use strict";

    var logger = require("jira/util/logger");
    var NavigationSubgroup = require("jira/projects/sidebar/component/navigation-subgroup");
    var SidebarComponent = require("jira/projects/sidebar/component");

    module("jira/projects/sidebar/component/navigation-subgroup", {
        markup: [
            '<li>',
            '   <a href="#" class="aui-nav-subtree-toggle"></a>',
            '   <a href="#" data-link-id="my-navigation-subgroup">Subgroup</a>',
            '   <ul>',
            '      <li><a class="aui-nav-item" data-link-id="my-navigation-item-1"></a></li>',
            '      <li><a class="aui-nav-item"></a></li>',
            '      <li><a class="aui-nav-item" data-link-id="my-navigation-item-3"></a></li>',
            '   </ul>',
            '</li>'
        ].join(''),

        buildNavigationSubgroup: function ()
        {
            return new NavigationSubgroup({
                el: this.markup
            });
        },

        buildNavigationSubgroupWithDuplicatedIDs: function ()
        {
            return new NavigationSubgroup({
                el: this.markup.replace("my-navigation-item-3", "my-navigation-item-1")
            });
        },

        triggerEventWithObject: function(emitter, event) {
            var eventObject = {
                isPrevented: false,
                emitter: emitter,
                preventDefault: function () {
                    this.isPrevented = true;
                }
            };
            emitter.trigger(event, eventObject);
            return eventObject;
        },

        assertEventRetriggered: function(navigationSubgroup, emitter, event) {
            var eventHandler = this.spy();
            navigationSubgroup.on(event, eventHandler);

            var eventObject = this.triggerEventWithObject(emitter, event);

            sinon.assert.calledOnce(eventHandler);
            sinon.assert.calledWith(eventHandler, eventObject);
        },

        assertEventPreventedAndRetriggered: function(navigationSubgroup, emitter, event) {
            var handler = function(ev){ev.preventDefault();};
            emitter.on(event, handler);

            var eventHandler = this.spy();
            navigationSubgroup.on(event, eventHandler);

            var eventObject = this.triggerEventWithObject(emitter, event);

            sinon.assert.calledOnce(eventHandler);
            sinon.assert.calledWith(eventHandler, eventObject);
            ok(eventObject.isPrevented);
            emitter.off(event, handler);
        },

        select: function(subgroup) {
            subgroup.$el.addClass("aui-nav-selected");
        }
    });

    test("When constructed, it extracts the link ID from the markup", function () {
        var navigationSubgroup = this.buildNavigationSubgroup();

        equal(navigationSubgroup.id, 'my-navigation-subgroup');
    });

    test("When constructed, it extracts the NavigationItems and NavigationSubgroups from the markup", function () {
        var navigationSubgroup = this.buildNavigationSubgroup();

        ok (navigationSubgroup.getItemAt(0) instanceof SidebarComponent.NavigationItem, "Item #0 is a NavigationItem");
        ok (navigationSubgroup.getItemAt(1) instanceof SidebarComponent.NavigationItem, "Item #1 is a NavigationItem");
        ok (navigationSubgroup.getItemAt(2) instanceof SidebarComponent.NavigationItem, "Item #2 is a NavigationItem");
    });

    test("When constructed, it lots a warning if the markup has duplicated IDs for groups", function() {
        var warnStub = this.stub(logger,'warn');
        this.buildNavigationSubgroupWithDuplicatedIDs();

        sinon.assert.calledOnce(warnStub);
        sinon.assert.calledWith(warnStub, "Duplicated IDs detected. There are more than one NavigationItem with id data-link-id='my-navigation-item-1'");
    });

    test("It deselects all the items", function () {
        var deselectStub = this.stub(SidebarComponent.NavigationItem.prototype, "deselect");
        var navigationSubgroup = this.buildNavigationSubgroup();

        navigationSubgroup.deselect();

        sinon.assert.calledOn(deselectStub, navigationSubgroup.getItemAt(0));
        sinon.assert.calledOn(deselectStub, navigationSubgroup.getItemAt(1));
        sinon.assert.calledOn(deselectStub, navigationSubgroup.getItemAt(2));
    });

    test("It fetches the items by the id", function () {
        var navigationSubgroup = this.buildNavigationSubgroup();

        ok(navigationSubgroup.getItem("my-navigation-item-1") instanceof SidebarComponent.NavigationItem, "Item #1 is a NavigationItem");
        ok(navigationSubgroup.getItem("my-navigation-item-3") instanceof SidebarComponent.NavigationItem, "Item #2 is a NavigationItem");
    });

    test("It fetches the items by the index", function () {
        var navigationSubgroup = this.buildNavigationSubgroup();

        ok(navigationSubgroup.getItemAt(0) instanceof SidebarComponent.NavigationItem, "Item #0 is a NavigationItem");
        ok(navigationSubgroup.getItemAt(1) instanceof SidebarComponent.NavigationItem, "Item #1 is a NavigationItem");
        ok(navigationSubgroup.getItemAt(2) instanceof SidebarComponent.NavigationItem, "Item #2 is a NavigationItem");
    });

    test("When a children triggers an event, the navigationSubgroup re-triggers it with the same EventObject", function () {
        var navigationSubgroup = this.buildNavigationSubgroup();

        var navigationItem = navigationSubgroup.getItemAt(0);
        this.assertEventRetriggered(navigationSubgroup, navigationItem, "before:select");
        this.assertEventRetriggered(navigationSubgroup, navigationItem, "select");
        this.assertEventRetriggered(navigationSubgroup, navigationItem, "before:deselect");
        this.assertEventRetriggered(navigationSubgroup, navigationItem, "deselect");
        this.assertEventRetriggered(navigationSubgroup, navigationItem, "before:navigate");
    });

    test("When a children triggers an event, the navigationSubgroup re-triggers it with the same EventObject, even if the original event was prevented", function () {
        var navigationSubgroup = this.buildNavigationSubgroup();

        var navigationItem = navigationSubgroup.getItemAt(0);
        this.assertEventPreventedAndRetriggered(navigationSubgroup, navigationItem, "before:select");
        this.assertEventPreventedAndRetriggered(navigationSubgroup, navigationItem, "before:deselect");
        this.assertEventPreventedAndRetriggered(navigationSubgroup, navigationItem, "before:navigate");
    });

    test("When a children triggers the before:select event, it deselects all the other items in the group", function () {
        var deselectItem = this.stub(SidebarComponent.NavigationItem.prototype, "deselect");
        var navigationSubgroup = this.buildNavigationSubgroup();

        this.triggerEventWithObject(navigationSubgroup.getItemAt(0), "before:select");

        sinon.assert.calledOn(deselectItem, navigationSubgroup.getItemAt(0));
        sinon.assert.calledOn(deselectItem, navigationSubgroup.getItemAt(1));
        sinon.assert.calledOn(deselectItem, navigationSubgroup.getItemAt(2));
    });

    test("When a children is selected, the item is not selected if the previously selected child prevents the 'before:deselect' event", function () {
        var navigationSubgroup = this.buildNavigationSubgroup();
        navigationSubgroup.getItemAt(0).select();
        navigationSubgroup.getItemAt(0).on("before:deselect", function(ev){ev.preventDefault();});

        navigationSubgroup.getItemAt(1).select();

        ok(navigationSubgroup.getItemAt(0).isSelected(), "Item #0 is still selected");
        ok(!navigationSubgroup.getItemAt(1).isSelected(), "Item #1 is not selected");
    });

    test("When deselecting all the items, it returns true if all the views were deselected", function () {
        var navigationSubgroup = this.buildNavigationSubgroup();

        var result = navigationSubgroup.deselect();

        strictEqual(result, true);
    });

    test("When deselecting all the items, it returns false if any 'before:deselect' event is cancelled", function () {
        var navigationSubgroup = this.buildNavigationSubgroup();
        navigationSubgroup.getItemAt(0).select();
        navigationSubgroup.getItemAt(0).on("before:deselect", function(ev){ev.preventDefault();});

        var result = navigationSubgroup.deselect();

        strictEqual(result, false);
    });

    test("When an item triggers the before:select event, it expands the subgroup", function() {
        var navigationSubgroup = this.buildNavigationSubgroup();
        ok(!navigationSubgroup.isExpanded());

        this.triggerEventWithObject(navigationSubgroup.getItemAt(0), "before:select");
        ok(navigationSubgroup.isExpanded());
    });

    test("Returns the expected ID when the getId method is called", function() {
        var navigationSubgroup = this.buildNavigationSubgroup();

        equal(navigationSubgroup.getId(), "my-navigation-subgroup");
    });

    test("Returns itself if it is selected and the getSelectedNavigationItem method is called", function() {
        var navigationSubgroup = this.buildNavigationSubgroup();
        this.select(navigationSubgroup);

        var selectedItem = navigationSubgroup.getSelectedNavigationItem();

        ok(selectedItem === navigationSubgroup);
    });

    test("Returns undefined if it is not selected and none if its children are selected and the getSelectedNavigationItem method is called", function() {
        var navigationSubgroup = this.buildNavigationSubgroup();

        var selectedItem = navigationSubgroup.getSelectedNavigationItem();

        ok(selectedItem === undefined);
    });

    test("Returns the selected child navigation item if the getSelectedNavigationItem method is called", function() {
        var navigationSubgroup = this.buildNavigationSubgroup();
        var itemToSelect = navigationSubgroup.getItemAt(0);
        itemToSelect.select();

        var selectedItem = navigationSubgroup.getSelectedNavigationItem();

        ok(selectedItem === itemToSelect);
    });

    test("Returns true if it is selected and the hasASelectedItem method is called", function() {
        var navigationSubgroup = this.buildNavigationSubgroup();
        this.select(navigationSubgroup);

        ok(navigationSubgroup.hasASelectedItem());
    });

    test("Returns true if it has a selected child and the hasASelectedItem method is called", function() {
        var navigationSubgroup = this.buildNavigationSubgroup();
        var itemToSelect = navigationSubgroup.getItemAt(0);
        itemToSelect.select();

        ok(navigationSubgroup.hasASelectedItem());
    });

    test("Returns false if it is not selected and none of its children are selected the hasASelectedItem method is called", function() {
        var navigationSubgroup = this.buildNavigationSubgroup();

        ok(!navigationSubgroup.hasASelectedItem());
    });

    test("Events are unbound when it is destroyed", function() {
        var view = this.buildNavigationSubgroup();
        var listener = sinon.stub();
        view.on('test-event', listener);

        view.destroy();
        view.trigger('test-event');

        sinon.assert.notCalled(listener);
    });
});

