define("jira/projects/sidebar/component/navigation-group", [
    "jira/projects/sidebar/component/navigation-group-factory",
    "jira/projects/libs/marionette",
    "jira/util/logger"
], function (NavGroupFactory, Marionette, logger) {
    "use strict";

    return Marionette.CompositeView.extend({
        childEvents: {
            "before:select": function (item, itemEvent) {
                this.retriggerPreventable("before:select", itemEvent);
                if (itemEvent.isPrevented) {
                    return;
                }

                // Deselect all the items. If they can't be deselected (i.e. someone prevented the before:deselect
                // event), force the before:select event to be prevented as well. This is useful in case someone else
                // is listening for this event.
                var itemWasDeselected = this.deselect();
                if (!itemWasDeselected) {
                    itemEvent.preventDefault();
                }
            },
            "select": function(child, ev) {
                this.trigger("select", ev);
            },
            "before:deselect": function(item, itemEvent) {
                this.retriggerPreventable("before:deselect", itemEvent);
            },
            "deselect": function(child, ev){
                this.trigger("deselect", ev);
            },
            "before:navigate": function(item, itemEvent) {
                this.retriggerPreventable("before:navigate", itemEvent);
            },
            "before:navigate:prevented": function(item, itemEvent) {
                this.trigger("before:navigate:prevented", itemEvent);
            }
        },

        initialize: function () {
            this.id = this.$el.attr("data-id");

            this.$(">ul>li").each(function (index, li) {
                var navigationItem = NavGroupFactory.build(li);

                var alreadyExists = !!this.getItem(navigationItem.id);
                if (alreadyExists) {
                    logger.warn("Duplicated IDs detected. There are more than one NavigationItem with id data-link-id='"+navigationItem.id+"'");
                }

                this.proxyChildEvents(navigationItem);
                this.children.add(navigationItem, navigationItem.id);
            }.bind(this));
        },

        onDestroy: function() {
            // This should be Marionette's responsibility, but it doesn't seem to ever get called (this might be a bug
            // in Marionette.
            this.unbind();
        },

        /**
         * Returns HTML element representing sidebar group.
         *
         * @returns {HTMLElement} HTML element representing sidebar group.
         */
        getElement: function() {
            return this.el;
        },

        /**
         * Deselect all the subviews.
         *
         * This method will call 'deselect()' for each item view.
         *
         * @returns {boolean} True if all the views where successfully deselected, false otherwise.
         */
        deselect: function () {
            var groupEvent = this.triggerPreventable("before:deselect");
            if (groupEvent.isPrevented) {
                return false;
            }

            // Can't use _.every() because it will stop the loop when the first 'false' value is found, and we still
            // want to call .deselect() in the rest of the views.
            var isSuccessful = true;
            this.children.each(function(view){
                isSuccessful = view.deselect() && isSuccessful;
            });
            return isSuccessful;
        },

        /**
         * Gets an item from the group.
         *
         * @param {string} id ID of the item to retrieve.
         * @returns {JIRA.Projects.Sidebar.Component.NavigationItem|JIRA.Projects.Sidebar.Component.NavigationSubgroup|undefined}
         */
        getItem: function (id) {
            return this.children.findByCustom(id);
        },

        /**
         * Gets an item based on its position.
         *
         * @param {number} index Position of the item.
         * @returns {JIRA.Projects.Sidebar.Component.NavigationItem|JIRA.Projects.Sidebar.Component.NavigationSubgroup|undefined}
         */
        getItemAt: function (index) {
            return this.children.findByIndex(index);
        },

        getSelectedNavigationItem: function() {
            var selectedItem = this.children.find(function(child) {
                return child.hasASelectedItem();
            });
            if (selectedItem) {
                return selectedItem.getSelectedNavigationItem();
            }
        },

        hasASelectedItem: function() {
            return this.children.any(function(child) {
                return child.hasASelectedItem();
            });
        }
    });
});
