define("jira/projects/release/submit-model", [
    "jira/util/logger",
    "jira/jquery/deferred",
    "backbone",
    "jira/projects/abstract-list/submit-model",
    "underscore",
    "jquery",
    "wrm/context-path",
    "jira/projects/release/submit-model-converter",
    "jira/projects/abstract-model/submit-model-states",
    "jira/ajs/ajax/ajax-util",
    "jira/projects/abstract-list/submit-model-events",
    "wrm/context-path",
    "jira/ajs/ajax/smart-ajax",
    "jira/util/events"
], function(
    logger,
    Deferred,
    Backbone,
    AbstractSubmitModel,
    _,
    $,
    contextPath,
    SubmitModelConverter,
    SubmitModelStates,
    ajaxUtil,
    SubmitModelEvents,
    wrmContextPath,
    SmartAjax,
    Events
) {
    var SubmitModel = AbstractSubmitModel.extend({
        RELATED_ISSUES_PATH: "/relatedIssueCounts",
        MERGE_PATH: "/mergeto/",

        urlRoot: contextPath() + "/rest/api/2/version",

        defaults: _.extend({}, AbstractSubmitModel.prototype.defaults, {
            released: false,
            archived: false,
            operations: []
        }),

        detailsPath: function() {
            return contextPath() + '/rest/projects/1.0/project/' + encodeURIComponent(this.get("project")) + '/release/details/' + this.id;
        },

        toDataJSON: function() {
            var json = this.toJSON();
            var data = _.pick(json, "project", "name", "description");
            data.userStartDate = json.startDate.formatted;
            data.userReleaseDate = json.releaseDate.formatted;

            return data;
        },

        _traceItemAdded: function () {
            logger.trace("jira.version.add.finished");
        },

        parse: function (resp, options) {
            var data = Backbone.Model.prototype.parse.call(this, resp, options);

            data.startDate = SubmitModelConverter.getDate(resp.userStartDate, resp.startDate);
            data.releaseDate = SubmitModelConverter.getDate(resp.userReleaseDate, resp.releaseDate);
            data.status = SubmitModelConverter.getDefaultStatus();

            delete data.userStartDate;
            delete data.userReleaseDate;

            return data;
        },

        destroy: function(options) {
            options = options || {};
            this.lastDestroyOptions = _.clone(options);
            var moveFixIssuesTo = (options.data) ? options.data.moveFixIssuesTo : undefined;
            var collection = this.collection;

            // do not remove model immediately, wait for REST result
            options.wait = true;

            //if model isNew then destroy returns false
            var result = Backbone.Model.prototype.destroy.apply(this, arguments);
            if (result) {
                result.fail(function (xhr) {
                    this.set("state", SubmitModelStates.ERROR_DELETE);
                    this.set("errorMsg", AJS.escapeHtml(ajaxUtil.getErrorMessageFromXHR(xhr)));
                }.bind(this));

                if (moveFixIssuesTo > 0) {
                    result.then(function () {
                        var targetCmp = collection.get(moveFixIssuesTo);
                        if (targetCmp) {
                            targetCmp.detailsSync().always(function () {
                                logger.trace("jira.version.delete.finished");
                            });
                        } else {
                            logger.trace("jira.version.delete.finished");
                        }
                    }.bind(this));
                } else {
                    result.always(function () {
                        logger.trace("jira.version.delete.finished");
                    });
                }

                this.trigger("removeItemEvent");
            }

            return result;
        },

        sync: function(method, model, options) {
            switch(method) {
                case 'delete':
                    var replacementData = this._convertFormToReplacementData(options.data);
                    options.attrs = replacementData;
                    delete options.data;
                    options.url = this.url() + "/removeAndSwap";
                    return Backbone.sync('create', model, options);
                case 'move':
                    options.url = this.url() + "/move";
                    options.attrs = model ? { after : model.url() } : { position: 'First' };
                    return Backbone.sync('create', this, options);
            }
            return Backbone.sync.apply(this, arguments);
        },

        _convertFormToReplacementData: function (formData) {
            var formData = formData || {};
            var replacementData = {};
            if (formData.affects === "swap") {
                replacementData.moveAffectedIssuesTo = formData.moveAffectedIssuesTo;
            }
            if (formData.fix === "swap") {
                replacementData.moveFixIssuesTo = formData.moveFixIssuesTo;
            }
            for (var dataName in formData) {
                var customFieldMatch = dataName.match(/^customField([0-9]+)Action$/);
                if (customFieldMatch) {
                    replacementData.customFieldReplacementList = replacementData.customFieldReplacementList || [];
                    var cfAction = formData[dataName];
                    var cfId = customFieldMatch[1];
                    var customFieldReplaceInfo = {
                        customFieldId: Number(cfId),
                        moveTo: cfAction === 'swap' ? (formData['customField' + cfId + 'MoveTo']) : null
                    };

                    replacementData.customFieldReplacementList.push(customFieldReplaceInfo);
                }
            }
            return replacementData;
        },

        /**
         * Overrides default save handler to only save (send to server) attributes that have changed.
         * Also provides some default error handling.
         *
         * @override
         * @param attributes
         * @param options
         */
        save: function (attributes, options) {
            this.lastSaveAttributes = _.clone(attributes);
            options = options || {};

            // if it is a new model, we don't have to worry about updating only changed attributes because they are all new
            if (this.isNew()) {
                // call super
                return AbstractSubmitModel.prototype.save.call(this, attributes, options);
            } else if (attributes) {
                var deferred = new Deferred();

                this.set("state", SubmitModelStates.isStateError(this.get("state")) ? SubmitModelStates.IN_PROGRESS_ERROR : SubmitModelStates.IN_PROGRESS);

                var error = options.error;
                var success = options.success;

                Backbone.Model.prototype.save.call(this, [], {
                    attrs: attributes
                }).done(function (xhr) {
                        this.detailsSync().always(function () {
                            this.set("state", SubmitModelStates.SUCCESSFUL);
                            deferred.resolve();
                            if (success) {
                                success.call(this, this, xhr);
                            }
                        }.bind(this));
                    }.bind(this))
                    .fail(function (xhr) {
                        this.set("state", SubmitModelStates.ERROR_UPDATE);
                        deferred.reject(xhr);
                        if( error ) {
                            var data = $.parseJSON(xhr.responseText || xhr.data);
                            error.call(this, this, data, xhr);
                        }
                    }.bind(this));

                return deferred;
            }
        },

        /**
         * Gets count for unresolved issues in this version
         *
         * @param options
         * ... {function} success - Server success callback
         * ... {function} error - Server error callback
         * @return JIRA.VersionModel
         */
        getUnresolvedIssueCount: function (options) {
            var versionId = this.id;
            options = options || {};
            var error = options.error;

            SmartAjax.makeRequest({
                url: wrmContextPath() + "/rest/api/2/version/" + versionId + "/unresolvedIssueCount",
                complete: function (xhr, status, smartAjaxResponse) {
                    if (smartAjaxResponse.successful) {
                        options.success.call(this, smartAjaxResponse.data);
                    } else {
                        // versionsDialog does not have error handler. Error will be shown on submit.
                        // we have to show the dialog or there will be only spinner displayed.
                        options.success.call(this, smartAjaxResponse.data);
                    }
                }.bind(this)
            });

            return this;
        },

        /**
         * Gets JSON representation of available versions to migrate issues of this version into.
         *
         * @return {Array}
         */
        getMoveVersionsJSON: function () {

            var moveVersions = this.collection
                .filter(function (model) {
                    return !model.get("released") && model !== this;
                }.bind(this))
                .map(function (model, i) {
                    var json = model.toJSON();

                    if (this.collection.at(i + 1) === this) {
                        json.nextScheduled = true;
                    }
                    return json;
                }.bind(this));
            return moveVersions;
        },

        /**
         * Gets count for issues with either affects version or fix version containing this version
         *
         * @param options
         * ... {function} success - Server success callback
         * ... {function} error - Server error callback
         * @return SubmitModel
         */
        getRelatedIssueCount: function (options) {
            var instance = this;

            options = options || {};

            SmartAjax.makeRequest({
                url: this.url() + this.RELATED_ISSUES_PATH,
                complete: function (xhr, status, smartAjaxResponse) {
                    if (smartAjaxResponse.successful) {
                        options.success.call(instance, smartAjaxResponse.data);
                    } else {
                        instance._serverErrorHandler(smartAjaxResponse);
                    }
                }
            });

            return this;
        },

        /**
         * Gets JSON representation of available versions to migrate issues of this version into.
         *
         * @return {Array}
         */
        getSwapVersionsJSON: function () {
            var instance = this,
                availableSwapVersions = [];

            this.collection.each(function (model) {
                if (!model.get("archived") && model !== instance) {
                    availableSwapVersions.push(model.toJSON());
                }
            });

            return availableSwapVersions.reverse();
        },

        /**
         * Throws a server error event unless user input validation error (status 400)
         *
         * @param xhr
         */
        _serverErrorHandler: function (xhr, ajaxOptions) {
            var data = {};
            if (xhr.status !== 400) {
                try {
                    data = $.parseJSON(xhr.responseText || xhr.data);
                } catch (e) {
                    data.errorMessages = [e];
                }
                Events.trigger(SubmitModelEvents.SERVER_ERROR, [data, xhr, ajaxOptions, this]);
            }
        },

        /**
         * Merges versions on the server
         *
         * @override
         * @param options
         * ... {function} success - Server success callback
         * ... {function} error - Server error callback
         * ... {object} data
         * ... ... {String} mergeTo - Merge target, all issues that have assigned this version will have version
         *  mergeTo assigned after merge
         *
         * @return SubmitModel
         */
        merge: function (options) {
            var instance = this,
                url = this.url(),
                data = options.data;
            var collection = this.collection;

            if (typeof data.mergeTo !== "string") {
                options.error.call(this, {status: 400, errorMessages: ["parameters"]});
            }

            SmartAjax.makeRequest({
                url: url + this.MERGE_PATH + data.mergeTo,
                type: "PUT",
                dataType: "json",
                timeout: 120000,

                complete: function (xhr, status, smartAjaxResponse) {
                    var smartAjaxResponseData = smartAjaxResponse.data;
                    if (typeof smartAjaxResponse.data === "string") {
                        try {
                            smartAjaxResponseData = JSON.parse(smartAjaxResponse.data);
                        } catch (e) {
                            smartAjaxResponseData = {
                                successful: false,
                                errorMessages: [String.valueOf(e)]
                            };
                            smartAjaxResponse.data = JSON.stringify(smartAjaxResponseData);
                        }
                    }

                    var isValidationError = !(xhr.status === 400 && smartAjaxResponseData && smartAjaxResponseData.errors);
                    if (smartAjaxResponse.successful) {
                        instance.trigger('destroy', instance, instance.collection, options);

                        if (options.success) {
                            options.success.call(instance, smartAjaxResponseData);
                        }
                        var destinationModel = collection.get(data.mergeTo);
                        if (destinationModel) {
                            destinationModel.detailsSync().always(function () {
                                destinationModel.set("state", SubmitModelStates.SUCCESSFUL);
                                destinationModel.addStateTimeout(SubmitModelStates.SUCCESSFUL, SubmitModelStates.READY);
                                logger.trace("jira.version.merge.finished");
                            });
                        } else {
                            logger.trace("jira.version.merge.finished");
                        }
                    } else if (isValidationError) {
                        instance._serverErrorHandler(smartAjaxResponse);
                        if (options.error) {
                            options.error.call(instance, smartAjaxResponseData);
                        }
                    }
                    if (options.complete) {
                        options.complete.call(instance, xhr.status, smartAjaxResponseData);
                    }
                }
            });

            return this;
        }
    });

    return SubmitModel;
});
