define("jira/projects/abstract-list/lazy-loading-collection", [
    "backbone"
], function (Backbone) {
    "use strict";

    /**
     * Collection which is able to load another set of items via calling 'nextPage()'.
     * Elements optionally can be filtered using 'this.filter.elementMatchFilter' method.
     * 
     * Current implementation assumes that all elements are pre-loaded and contained under 'this.originalCollection'.
     * All elements applicable to selected filter are contained under 'this.fullCollection'.
     * Currently visible elements are just accessible as regular collection.
     *  
     */
    var LazyLoadingCollection = Backbone.Collection.extend({
        collectionClass: null,
        _findAddingIndex: function (collection, addingElement) {
            return 0;
        },

        initialize: function initialize(items, options) {
            this.originalCollection = new this.collectionClass(options.data);
            this.fullCollection = new this.collectionClass(this.originalCollection.models);

            this.nextPage();
            this.listenTo(this, "remove", function () {
                if (this.originalCollection.length <= 1) {
                    this.trigger("collectionChanged");
                }
            });
        },

        nextPage: function () {
            if (this.isFullyLoaded()) {
                return false;
            }
            this.add(this.fullCollection.slice(this.size(), Math.min(this.size() + LazyLoadingCollection.PAGE_INCREMENT, this.fullCollection.size())));

            return !this.isFullyLoaded();
        },

        isFullyLoaded: function isFullyLoaded() {
            return this.size() >= this.fullCollection.size();
        },

        updateElement: function(element) {
            var id = element.get("id");
            var elementJson = element.toJSON();

            var model = this.fullCollection.get(id);
            if (model) {
                model.set(elementJson);
            }

            this.originalCollection.filter(function(originalElement) {
                return originalElement.id == id;
            }).some(function(originalElement) {
                originalElement.set(elementJson);
            });

            return model;
        },

        addElement: function (addedElement) {
            this.originalCollection.add(addedElement, {at: this._findAddingIndex(this.originalCollection, addedElement)});

            if (this.filter.elementMatchFilter(addedElement)) {
                var index = this._findAddingIndex(this.fullCollection, addedElement);

                this.fullCollection.add(addedElement, {at: index});

                if (index <= this.size()) {
                    this.add(addedElement, {at: index});
                }
            }

            if (this.size() <= 1) {
                this.trigger("update");
            }
            this.trigger("collectionChanged");

        },

        resetFilter: function(filteredElements) {
            this.fullCollection.reset(filteredElements);

            this.reset();
            this.nextPage();
        },

        elementsLeft: function() {
            return {
                elementsTotal: this.fullCollection.size(),
                elementsLoaded: this.size(),
                elementsLeft: this.fullCollection.size() - this.size()
            };
        }
    });

    LazyLoadingCollection.PAGE_INCREMENT = 25;

    return LazyLoadingCollection;
});