define("jira-project-config/workflow/entities/manager", ["require"], function (require) {
    "use strict";

    var jQueryDeferred = require('jira/jquery/deferred');
    var jQuery = require('jquery');
    var _ = require('underscore');
    var entityParser = require('jira-project-config/issuetypes/entities/entityParser');
    var ProjectWorkflowManager = require('jira-project-config/workflow/entities/ajax/project-workflow-manager');

    /**
     * Describes an error when retrieving an entity.
     *
     * @typedef {object} JIRA.ProjectConfig.IssueTypes.Entities.ManagerError
     * @property {string} type A classification of the error, one of: "error", "abort", "unauthenticated".
     * @property {string} [message] A description of the error, suitable for presenting to the user.
     */

    /**
     * Translates a failed jqXHR response into an an error suitable for returning from a manager.
     * @param jqXHR A jQuery jqXHR representing the response.
     * @returns {JIRA.ProjectConfig.IssueTypes.Entities.ManagerError}
     */
    function makeError(jqXHR) {
        var statusText = jqXHR.statusText;
        var statusCode = jqXHR.status;
        var error = {};

        if (statusText === "abort") {
            error.type = "abort";
        } else if (statusCode === 401) {
            error.type = "unauthenticated";
        } else {
            error.type = "error";
        }

        if (_.contains([400, 403, 404], statusCode) && jqXHR.responseText) {
            try {
                // Handle serialised ErrorCollections.
                error.message = JSON.parse(jqXHR.responseText).errorMessages[0];
            } catch (e) {
                error.message = jqXHR.responseText;
            }
        }

        return error;
    }

    /**
     * @class
     * @classdesc An API for retrieving entities.
     * @constructor
     * @param {Marionette.RequestResponse} options.reqres A reqres that must respond to `urls:issueTypes:*`
     * @param {JIRA.ProjectConfig.IssueTypes.Ajax.ProjectIssueTypeManager} [options.projectIssueTypeAjaxManager] The
     *     AJAX manager for the project issue type REST APIs.
     */
    var Manager = function Manager(options) {
        _.defaults(options, {
            projectWorkflowManager: new ProjectWorkflowManager({
                reqres: options.reqres
            })
        });

        _.extend(this, _.pick(options, "reqres", "projectWorkflowManager"));
    };

    /** @lends JIRA.ProjectConfig.Workflow.Entities.Manager# */
    Manager.prototype = {
        /**
         * Retrieve the workflow for an issue type in a project.
         *
         * @param {string} options.projectKey
         * @param {string} options.workflowName
         * @returns a jQuery promise that is resolved with a {@link JIRA.ProjectConfig.IssueTypes.Entities.Workflow},
         *   or rejected with an {@link JIRA.ProjectConfig.IssueTypes.ManagerError}
         */
        workflowData: function workflowData(options) {
            var wish = jQueryDeferred();
            var fetchingWorkflowData = this.projectWorkflowManager.workflow(options.projectKey, options.workflowName);
            var fetchingIssueTypes = this.reqres.request("issueTypeLinks");

            jQuery.when(fetchingWorkflowData, fetchingIssueTypes).done(function (workflowData, issueTypeLinks) {
                var workflow = entityParser.makeWorkflow(workflowData, issueTypeLinks);
                wish.resolve(workflow);
            })
            // Only fetchingWorkflowData can fail, so we can safely assume that any problems are due to it, this may
            // change in the future.
            .fail(_.compose(wish.reject, makeError));

            return wish.promise();
        }
    };

    return Manager;
});