define("jira-project-config/workflow/controller", ["require"], function (require) {
    "use strict";

    var $ = require('jquery');
    var Marionette = require('jira-project-config/marionette');
    var _ = require('underscore');
    var navigate = require('jira-project-config/navigate');
    var formatter = require('jira/util/formatter');
    var parseUri = require('jira/libs/parse-uri');

    var WorkflowView = require('jira-project-config/workflow/view');
    var ProgressIndicatorView = require('jira-project-config/utils/widgets/progress-indicator');

    var IssueType = require('jira-project-config/issuetypes/entities/models/issue-type');
    var HeaderController = require('jira-project-config/issuetypes/header/controller');
    var WorkflowController = require('jira-project-config/issuetypes/perspectives/workflow/controller');

    return Marionette.Controller.extend(
    /** @lends JIRA.ProjectConfig.Workflow.Controller# */
    {
        /**
         * The events and corresponding functions that handle each event on the internal vent.
         */
        internalEvents: {
            "workflow:publishComplete workflow:discardComplete": "_onPublishDiscardComplete",
            "workflow:editWorkflow": "_onEditWorkflow",
            "workflow:exitDraft": "_onExitDraft"
        },

        /**
         * The route functions that should be wrapped with deferred wrappers.
         */
        deferredRoutes: ["editWorkflow", "viewWorkflow"],

        /**
         * The route functions that should be wrapped with rendering wrappers.
         */
        renderingRoutes: ["editWorkflow", "viewWorkflow"],

        /**
         * @classdesc The top level controller for the Issue Types section in Project Administration.
         *
         * This controller is responsible for orchestrating the behaviour of the page by deferring to sub controllers.
         * @constructs
         * @extends Marionette.Controller
         * @param {JIRA.ProjectConfig.Application} options.application The shared application.
         * @param {JIRA.ProjectConfig.IssueTypes.Model} options.model A model representing the high level state of the page.
         * @param {Marionette.Region} options.region The region to render within.
         */
        initialize: function initialize(options) {
            this.application = options.application;
            this.model = options.model;
            this.view = new WorkflowView({
                region: options.region
            });

            this.request = 0;
            this.deferredRoutesExecuting = 0;
            this.internalVent = new Marionette.Wreqr.EventAggregator();
            this.progressIndicatorView = new ProgressIndicatorView();

            this._initializeEventHandlers();
            this._initializeDeferredRouteWrappers();
            this._initializeRenderingRouteWrappers();
        },

        viewWorkflowRoute: function viewWorkflowRoute(projectKey) {
            this._handleRoute(this.viewWorkflow, projectKey);
        },
        editWorkflowRoute: function editWorkflowRoute(projectKey) {
            this._handleRoute(this.editWorkflow, projectKey);
        },
        _handleRoute: function _handleRoute(route, projectKey) {
            var workflowName = decodeURIComponent(parseUri(window.location.search).queryKey.name || '');
            if (workflowName) {
                route.call(this, projectKey, workflowName);
            } else {
                this.error404();
            }
        },

        /**
         * Handler for viewing an issue type's workflow.
         */
        viewWorkflow: function viewWorkflow(projectKey, workflowName) {
            var app = this.application;
            var request = ++this.request;

            var fetchingProject = this.application.request("project");
            var fetchingWorkflow = this.application.request("workflowData", {
                projectKey: projectKey,
                workflowName: workflowName
            });

            var fetchingWorkflowLayoutData = fetchingWorkflow.pipe(function (workflow) {
                return app.request("workflowLayoutData", workflow);
            });

            var isCurrent = function () {
                return request === this.request;
            }.bind(this);

            return $.when(fetchingProject, fetchingWorkflow, fetchingWorkflowLayoutData).done(function (project, workflow, workflowLayoutData) {
                if (!isCurrent()) {
                    return;
                }

                this._updateModel(project, workflow, false);

                this.workflowController.show(workflowLayoutData.layoutData);
            }.bind(this)).fail(function (error) {
                if (!isCurrent()) {
                    return;
                }
                this._handleError(error);
            }.bind(this));
        },

        /**
         * Handler for editing an issue type's workflow.
         */
        editWorkflow: function editWorkflow(projectKey, workflowName) {
            var app = this.application;
            var request = ++this.request;

            var fetchingProject = this.application.request("project");
            var fetchingWorkflow = this.application.request("workflowData", {
                projectKey: projectKey,
                workflowName: workflowName
            });

            var fetchingWorkflowLayoutData = fetchingWorkflow.pipe(function (workflow) {
                return app.request("workflowLayoutData", workflow, true);
            });

            var isCurrent = function () {
                return request === this.request;
            }.bind(this);

            return $.when(fetchingProject, fetchingWorkflow, fetchingWorkflowLayoutData).done(function (project, workflow, workflowLayoutData) {
                if (!isCurrent()) {
                    return;
                }

                var layoutData = workflowLayoutData.layoutData;
                if (layoutData && workflow.get('updatedDate') < layoutData.updatedDate.getTime()) {
                    workflow.set("isDraftWithChanges", true);
                }

                this._updateModel(project, workflow, true);

                this.workflowController.show(layoutData);
            }.bind(this)).fail(function (error) {
                if (!isCurrent()) {
                    return;
                }
                this._handleError(error);
            }.bind(this));
        },

        /**
         * Handler for when a requested page was not found.
         */
        error404: function error404() {
            this.application.execute("error:404", this.application.content);
        },

        /**
         * Displays the issue types view in the application's content region. Also resets the controllers with their respective
         * regions (as the region they reference is now stale). Shows the header.
         *
         * @private
         */
        _displayView: function _displayView() {
            this.application.content.show(this.view);

            this._initializeControllers();
            this.headerController.show();
        },

        /**
         * @private
         */
        _updateModel: function _updateModel(project, workflow, editing) {
            // compatibility with the Header & PerspectiveWorkflow
            var dummyIssueType = new IssueType({
                id: 0,
                name: '',
                workflow: workflow
            });

            this.model.set({
                project: project,
                workflow: workflow,
                selectedIssueType: dummyIssueType,
                editing: editing
            });
        },

        /**
         * @returns {string} the url
         * @private
         */
        _getViewWorkflowURL: function _getViewWorkflowURL() {
            return this.application.reqres.request("urls:workflows:viewWorkflow", {
                project: this.model.get('project'),
                workflow: this.model.get('workflow')
            });
        },

        /**
         * @returns {string} the url
         * @private
         */
        _getEditWorkflowURL: function _getEditWorkflowURL() {
            return this.application.reqres.request("urls:workflows:editWorkflow", {
                project: this.model.get('project'),
                workflow: this.model.get('workflow')
            });
        },

        /**
         * Do something appropriate with an error.
         * @param {JIRA.ProjectConfig.IssueTypes.Entities.ManagerError} error
         * @private
         */
        _handleError: function _handleError(error) {
            if (_.contains(["error", "abort"], error.type)) {
                this.application.commands.execute("error:generic", this.application.content, error.message || formatter.I18n.getText("admin.projectconfig.error.generic.message"));
            } else if (error.type === "unauthenticated") {
                navigate(this.application.reqres.request("urls:workflows:login", {
                    returnUrl: window.location.href
                }));
            }
        },

        /**
         * Initializes the header, fields and workflow controllers.
         * @private
         */
        _initializeControllers: function _initializeControllers() {
            this.workflowController = new WorkflowController({
                commands: this.application.commands,
                model: this.model,
                region: this.view.content,
                vent: this.internalVent
            });

            this.headerController = new HeaderController({
                commands: this.application.commands,
                model: this.model,
                perspectives: null,
                region: this.view.header,
                vent: this.internalVent
            });
        },

        /**
         * Initializes event listening for the events and corresponding methods declared in the `internalEvents` property.
         * @private
         */
        _initializeEventHandlers: function _initializeEventHandlers() {
            this.listenToOnce(this.view, "render", function () {
                this._maybeShowProgressIndicator(true);
            });

            _.each(this.internalEvents, function (handler, events) {
                this.listenTo(this.internalVent, events, _.bind(this[handler], this));
            }, this);

            this.listenTo(this.model, "change:workflow", function (model, workflow) {
                var name = workflow ? workflow.get('displayName') : '';
                this.application.title(formatter.I18n.getText("admin.project.singleworkflow.title", name));
            });
        },

        /**
         * Wraps all functions declared in the `renderingRoutes` property with behaviour that needs to occur before (or after)
         * a rendering route method is executed.
         * @private
         */
        _initializeRenderingRouteWrappers: function _initializeRenderingRouteWrappers() {
            var instance = this;

            var getWrapper = function getWrapper(func) {
                var toExecute = instance[func];
                return function () {
                    if (instance.application.content.currentView !== instance.view) {
                        this._displayView();
                    }
                    instance.internalVent.trigger("before:perspectiveRerender");
                    toExecute.apply(instance, arguments);
                };
            };

            this._wrapFunctions(this.renderingRoutes, getWrapper);
        },

        /**
         * Wraps all functions declared in the `deferredRoutes` property with behaviour that needs to occur before, or on completion
         * of the methods' returned deferreds.
         * @private
         */
        _initializeDeferredRouteWrappers: function _initializeDeferredRouteWrappers() {
            var instance = this;

            var getWrapper = function getWrapper(func) {
                var toExecute = instance[func];
                return function () {
                    instance._maybeShowProgressIndicator();
                    instance.deferredRoutesExecuting++;
                    toExecute.apply(instance, arguments).always(function () {
                        instance.deferredRoutesExecuting--;
                        if (instance.deferredRoutesExecuting === 0) {
                            instance._closeProgressIndicatorIfPresent();
                        }
                    });
                };
            };

            this._wrapFunctions(this.deferredRoutes, getWrapper);
        },

        _wrapFunctions: function _wrapFunctions(functions, getWrapper) {
            _.each(functions, function (func) {
                this[func] = getWrapper(func);
            }, this);
        },

        /**
         * Sets a timeout for showing the progress indicator if a timeout does not already exist.
         * @param {boolean} always show the progress indicator
         * @private
         */
        _maybeShowProgressIndicator: function _maybeShowProgressIndicator(always) {
            var show = function (instance) {
                return function () {
                    instance.view.progressIndicator.show(instance.progressIndicatorView);
                };
            }(this);

            if (always) {
                show();
            } else if (!this._progressIndicatorTimeoutId) {
                this._progressIndicatorTimeoutId = setTimeout(function () {
                    show();
                }, 500);
            }
        },

        /**
         * If aa progress indicator timeout exists, it clears it and closes the progress indicator region.
         * @private
         */
        _closeProgressIndicatorIfPresent: function _closeProgressIndicatorIfPresent() {
            if (this._progressIndicatorTimeoutId) {
                clearTimeout(this._progressIndicatorTimeoutId);
                delete this._progressIndicatorTimeoutId;
            }
            this.view.progressIndicator && this.view.progressIndicator.close();
        },

        /**
         * @private
         */
        _onPublishDiscardComplete: function _onPublishDiscardComplete() {
            navigate(this._getViewWorkflowURL());
        },

        /**
         * @private
         */
        _onEditWorkflow: function _onEditWorkflow() {
            navigate(this._getEditWorkflowURL());
        },

        /**
         * @private
         */
        _onExitDraft: function _onExitDraft() {
            navigate(this._getViewWorkflowURL());
        }
    });
});