define("jira-project-config/screen-editor/view", ['jira/util/key-code', 'jira/util/formatter', 'jira/jquery/deferred', 'jquery', 'wrm/context-path', 'underscore', 'jira/dialog/form-dialog', 'aui/restful-table', "jira-project-config/backbone", 'jira-project-config/screen-editor/add-tab', 'jira-project-config/screen-editor/edit-tab', 'jira-project-config/screen-editor/template', 'jira-project-config/screen-editor/utils', 'jira/util/logger', 'jira/util/strings', 'aui/inline-dialog', 'jira-project-config/screen-editor/add-field-view'], function (keyCodes, formatter, Deferred, jQuery, wrmContextPath, _, FormDialog, RestfulTable, Backbone, ScreenAddTabView, ScreenEditTabView, TemplateScreenEditor, Utils, logger, strings, inlineDialog, AddFieldView) {
    "use strict";

    var inlineDialogWidth = 400;

    return Backbone.View.extend({
        initialize: function initialize() {

            // Whenever we add a new tab. We activate it.
            this.model.bind("tabAdded", function (tab) {
                this.render(tab);
                jQuery(this.el).trigger("tabAdded");
            }.bind(this));

            this.model.bind("tabRenamed", function (tab) {
                this.render(tab).done(function () {
                    jQuery(this.el).trigger("tabRenamed");
                }).always(function () {
                    logger.trace("tab.rename.complete");
                });
            }.bind(this));

            jQuery(document).keydown(function (e) {
                if (document.activeElement === document.body && e.keyCode === keyCodes.TAB) {
                    e.preventDefault();
                    if (e.shiftKey) {
                        this.prev();
                    } else {
                        this.next();
                    }
                }
            }.bind(this));
        },

        // Delegate events
        events: {
            "click .delete-tab": "deleteTab",
            "click .menu-item:not(.active-tab)": "activateTab"
        },

        next: function next() {
            var active = this.$el.find(".active-tab");
            var next = active.next(".menu-item");

            if (!next.length) {
                next = this.$el.find(".menu-item").first();
            }

            next.click();
        },

        prev: function prev() {
            var active = this.$el.find(".active-tab");
            var prev = active.prev(".menu-item");

            if (!prev.length) {
                prev = this.$el.find(".menu-item").last();
            }

            prev.click();
        },

        /**
         * Shows a confirmation dialog for deleting tab
         * @param {Event} e
         */
        deleteTab: function deleteTab(e) {

            var instance = this;
            var $tab = jQuery(e.currentTarget);
            var tab = $tab.data("tab");
            var dialog = new FormDialog({
                id: "delete-tab-" + tab,
                content: function content(ready) {
                    var tabName = $tab.closest(".menu-item").find(".tab-label").text();
                    tabName = strings.escapeHtml(tabName);
                    instance.model.getFieldsForTab(tab).done(function (fields) {
                        ready(jQuery(TemplateScreenEditor.deleteTabConfirmation({
                            name: tabName,
                            fields: fields
                        })));
                    });
                }
            });

            dialog._submitForm = _.bind(function (e) {
                e.preventDefault();
                this.model.deleteTab(tab).done(_.bind(function () {
                    jQuery(this.el).trigger("tabDeleted");
                    this.render(this.selectedTab);
                }, this));
                dialog.hide();
            }, this);

            dialog.show();
            e.stopPropagation(); // Stop aui tabs from activating tab
            e.preventDefault(); // Don't follow link
        },

        /**
         * Used for draggin between tabs. This will open the specified tab, add the field to it, then activate sorting on it.
         * @param {$} $tab - The tab to activate
         * @param {String} field - field id
         */
        activateTabWithField: function activateTabWithField($tab, field) {
            var instance = this;
            var tab = $tab.data("tab");
            var currentTab = this.$el.find(".active-tab").data("tab");
            this.model.addFieldToTab($tab.data("tab"), currentTab, field).done(_.bind(function () {
                instance.table.$tbody.sortable("destroy");
                jQuery(instance.el).trigger("fieldMovedTab");
                this.render(tab);
            }, this)).fail(function () {
                instance.table.$tbody.sortable("cancel");
            });
        },

        _hoverShimEventHandler: function _hoverShimEventHandler(event) {
            // When we upgrade to jQuery 1.9+ we will need to verify this continues to work.
            // The semantics around event.srcElement and event.target change in 1.9.
            // It's likely that we will need to use event.target instead.
            var $target = jQuery(event.srcElement);
            if (!$target.hasClass("hover-shim")) {
                // jQuery detects the incorrect target in FF; get it from the original event instead.
                $target = jQuery(event.originalEvent.target);
            }
            if ($target.hasClass("hover-shim")) {
                this.table.$tbody.sortable({
                    update: null
                });
                $target.trigger("drop", event);
            }
        },

        /**
         * When dragging between tabs, we position transparent <div>'s over the top of the tabs with a higher zIndex then the field
         * we are dragging. This is so that when the mouse hovers over the tab the mouse enter event is fired. We then wait a
         * little bit and if we are still on the tab we activate it, adding the field we are dragging.
         * @param {String} field
         */
        activateHoverShims: function activateHoverShims(field) {
            var instance = this;

            this.$el.find(".menu-item:not(.active-tab)").each(function () {
                var $tab = jQuery(this);
                $tab.addClass("field-drop-target");
                var offset = $tab.offset();
                jQuery("<div class='hover-shim'/>").css({
                    position: "absolute",
                    top: offset.top,
                    left: offset.left,
                    width: $tab.outerWidth(),
                    height: $tab.outerHeight(),
                    zIndex: 9999
                }).mouseenter(function () {
                    $tab.addClass("hover");
                }).mouseleave(function () {
                    $tab.removeClass("hover");
                }).on("drop", function () {
                    instance.activateTabWithField($tab, field);
                }).appendTo("body");
            });
        },

        /**
         * Removes all the hover shims
         */
        disableHoverShims: function disableHoverShims() {
            this.$el.find(".field-drop-target").removeClass("field-drop-target");
            jQuery(".hover-shim").each(function () {
                var $shim = jQuery(this);
                $shim.remove();
            });
        },

        /**
         * Activates the tab from $ element
         * @param {Event} e
         */
        activateTab: function activateTab(e) {
            this.render(jQuery(e.currentTarget).data("tab"));
            e.preventDefault();
            e.stopPropagation();
        },

        /**
         * Renders screen editor. Including tabs and restful table
         *
         * @param {Long} tab - Active tab id
         * @return {*}
         */
        render: function render(tab) {
            var deferred = Deferred();
            var instance = this;
            var model = this.model;
            var selectedTab = tab;
            var $el = this.$el;

            $el.addClass("screen-editor");
            $el.toggleClass("read-only", model.get("readOnly"));

            model.getTabs().done(function (tabs) {
                // If supplied tab doesn't exist make first tab selected
                if (!_.any(tabs, function (thisTab) {
                    return thisTab.id === parseInt(tab, 10);
                })) {
                    selectedTab = tabs[0].id;
                }

                // Renders the tabs
                $el.html(TemplateScreenEditor.tabs({
                    selectedTab: selectedTab,
                    tabs: tabs
                }));

                instance._renderTabs($el, tabs, selectedTab);
                instance._renderTable($el, selectedTab).done(deferred.resolve);

                if (window.sessionStorage) {
                    window.sessionStorage.setItem("selectedScreenTab", selectedTab);
                }
                instance.selectedTab = selectedTab;
            }).fail(function () {
                deferred.reject();
            });
            return deferred.promise();
        },

        _renderTabs: function _renderTabs($container, tabs, selectedTab) {
            var model = this.model;
            var readOnly = model.get("readOnly");
            // Renders the tabs
            $container.html(TemplateScreenEditor.tabs({
                selectedTab: selectedTab,
                tabs: tabs
            }));

            if (!readOnly) {
                // Setup sortable tabs
                var $tabMenu = jQuery(".tabs-menu", $container);
                $tabMenu.sortable({
                    delay: 0,
                    cursor: "move",
                    items: ".menu-item",
                    update: function update(event, ui) {
                        var index = jQuery(".tabs-menu .menu-item", $container).index(ui.item);
                        model.updateTabPosition(ui.item.data("tab"), index).done(function () {
                            logger.trace("screen.tab.order.updated");
                            $container.trigger("tabReordered");
                        }).fail(function () {
                            $tabMenu.sortable("cancel");
                        });
                    }
                });

                // Initialize the add tab dialog
                inlineDialog(".add-tab", "add-tab", function (contents, trigger, doShowPopup) {
                    var addTabView = new ScreenAddTabView({
                        model: model,
                        el: contents
                    });
                    addTabView.render();
                    doShowPopup();
                    _.defer(function () {
                        contents.find(":text").focus();
                    });
                }, {
                    width: inlineDialogWidth
                });

                // Initialize the edit tab dialog

                inlineDialog(jQuery(".edit-tab:visible"), "edit-tab", function (contents, trigger, doShowPopup) {
                    var editTabView = new ScreenEditTabView({
                        model: model,
                        tabs: tabs,
                        selectedTabId: selectedTab,
                        el: contents
                    });
                    editTabView.render();

                    doShowPopup();
                    _.defer(function () {
                        contents.find(":text").focus();
                    });
                }, {
                    width: inlineDialogWidth
                });
            }
        },

        _renderTable: function _renderTable($container, selectedTab) {
            var addFieldView;
            var deferred = Deferred();
            var instance = this;
            var model = this.model;
            var projectKey = model.get("projectKey");
            var readOnly = model.get("readOnly");
            var RowModel;
            var ViewRow;

            RowModel = RestfulTable.EntryModel.extend({
                save: function save(data, options) {
                    var oldError = options.error;
                    options.error = function (model, data, xhr) {

                        //If a 400 error comes back with no field related errors, then display a global
                        //error message because it will be ignored otherwise.
                        if (xhr.status === 400 && (!data || _.isEmpty(data.errors))) {
                            Utils.showErrorDialogForData(data);
                        }

                        if (oldError) {
                            return oldError.apply(this, arguments);
                        }
                    };
                    //JS === super(data, options);
                    return RestfulTable.EntryModel.prototype.save.call(this, data, options);
                }
            });

            ViewRow = RestfulTable.Row.extend({
                renderOperations: function renderOperations() {
                    if (!readOnly) {
                        return jQuery(TemplateScreenEditor.fieldRowOperations({
                            extraClasses: this.classNames.DELETE
                        })).click(function (e) {
                            e.preventDefault();
                            this.destroy();
                        }.bind(this));
                    }
                    return null;
                },
                renderDragHandle: function renderDragHandle() {
                    return TemplateScreenEditor.fieldRowReorder({
                        extraClasses: this.classNames.DRAG_HANDLE
                    });
                },
                destroy: function destroy() {
                    this.model.destroy({
                        error: function error(xhr) {
                            //Non 400 errors are processed by the AJS.RestfulTable.Events.SERVER_ERROR handler.
                            if (xhr.status === 400) {
                                Utils.showErrorDialogForXhr(xhr);
                            }
                        }
                    });
                }
            });

            // https://developer.atlassian.com/display/AUI/AJS.RestfulTable
            this.table = new RestfulTable({
                el: jQuery("#tab-" + selectedTab),
                allowCreate: false,
                allowEdit: false,
                allowReorder: !readOnly,
                createPosition: "bottom",
                resources: {
                    all: wrmContextPath() + "/rest/api/2/screens/" + model.get("id") + "/tabs/" + selectedTab + "/fields" + (projectKey ? "?projectKey=" + encodeURI(projectKey) : ""),
                    self: wrmContextPath() + "/rest/api/2/screens/" + model.get("id") + "/tabs/" + selectedTab + "/fields"
                },
                noEntriesMsg: formatter.I18n.getText('admin.screens.no.fields'),
                columns: [{
                    id: "name",
                    styleClass: "field-name-cell",
                    header: "Field"
                }, {
                    id: "type",
                    header: "Type"
                }],
                model: RowModel,
                views: {
                    row: ViewRow
                }
            });

            if (readOnly) {
                jQuery(this.table).on(RestfulTable.Events.INITIALIZED, function () {
                    instance.table.$el.removeClass("aui-restfultable-allowhover");
                });
            } else {
                // Modifying the default restfultable sortable behaviour
                this.table.$tbody.sortable({
                    axis: false,
                    appendTo: "body",
                    containment: false,
                    activate: function activate(event, ui) {
                        // change colspan to 5 (one column is not included)
                        ui.placeholder.find('td').attr('colspan', 5);
                    }
                });

                // Activating and disabling our hover shims on sortstart/end
                this.table.$tbody.on("sortstart", function (event, ui) {
                    if (jQuery.browser.msie && jQuery.browser.version === 8) {
                        ui.placeholder.height(ui.helper.height());
                    }
                    jQuery("body").addClass("sorting-fields");
                    instance.activateHoverShims(instance.table.getRowFromElement(ui.item[0]).model.id);
                }).on("sortstop", function () {
                    jQuery("body").removeClass("sorting-fields");
                    instance.disableHoverShims();
                }).on("sortbeforestop", _.bind(instance._hoverShimEventHandler, this));

                // When we remove a field we need to re render the fields list as it will now include
                // the remove field
                jQuery(this.table).on(RestfulTable.Events.ROW_REMOVED, function () {
                    $container.trigger("fieldRemoved");
                    addFieldView.fetchFieldsAndRerender();
                });

                jQuery(this.table).on(RestfulTable.Events.ROW_ADDED, function () {
                    $container.trigger("fieldAdded");
                });

                jQuery(this.table).on(RestfulTable.Events.REORDER_SUCCESS, function () {
                    $container.trigger("fieldReordered");
                });

                addFieldView = new AddFieldView({
                    el: jQuery(".add-field-container"),
                    model: model,
                    table: this.table,
                    selectedTab: selectedTab,
                    projectKey: projectKey
                });
                addFieldView.render();
                addFieldView.fetchFieldsAndRerender();
            }

            // Once the restful table has finished loading. So have we.
            jQuery(this.table).on(RestfulTable.Events.INITIALIZED, function () {
                deferred.resolve();
            });

            return deferred.promise();
        }
    });
});