define("jira-project-config/screen-editor/model", ['jira/jquery/deferred', 'jquery', 'wrm/context-path', 'underscore', "jira-project-config/backbone", 'jira-project-config/screen-editor/utils'], function (Deferred, jQuery, wrmContextPath, _, Backbone, Utils) {
    "use strict";

    var safeAjax = function safeAjax() {
        var def = jQuery.ajax.apply(jQuery, arguments);
        def.fail(Utils.showErrorDialogForXhr);
        return def;
    };

    /**
     * Screen editor model talks to the server.
     */
    return Backbone.Model.extend({

        _projectParam: function _projectParam() {
            var projectKey = this.get("projectKey");
            return projectKey ? "?projectKey=" + encodeURI(projectKey) : "";
        },

        /**
         * Gets all the tabs for screen
         * @return {$.Deferred}
         */
        getTabs: function getTabs() {
            return safeAjax({
                url: wrmContextPath() + "/rest/api/2/screens/" + this.get("id") + "/tabs" + this._projectParam()
            });
        },
        /**
         * Adds a tab to screen
         * @param {Number} tab - TabId
         * @param {String} field - field id
         * @return {$.Deferred}
         */
        addFieldToTab: function addFieldToTab(tab, currentTab, field) {
            var deferred = Deferred();
            var instance = this;
            safeAjax({
                url: wrmContextPath() + "/rest/api/2/screens/" + this.get("id") + "/tabs/" + currentTab + "/fields/" + field,
                type: "DELETE",
                contentType: "application/json"
            }).done(function () {
                safeAjax({
                    url: wrmContextPath() + "/rest/api/2/screens/" + instance.get("id") + "/tabs/" + tab + "/fields",
                    type: "POST",
                    contentType: "application/json",
                    dataType: "json",
                    data: JSON.stringify({
                        fieldId: field
                    })
                }).done(function () {
                    deferred.resolve();
                }).fail(function () {
                    deferred.reject();
                });
            }).fail(function () {
                deferred.reject();
            });
            return deferred.promise();
        },
        /**
         * Renames a tab
         * @param {Number} tab - tab id
         * @param {String} name - New name
         * @return {$.Deferred}
         */
        renameTab: function renameTab(tab, name) {
            //The errors returned on rename are handled in method that call this.
            return jQuery.ajax({
                type: "PUT",
                url: wrmContextPath() + "/rest/api/2/screens/" + this.get("id") + "/tabs/" + tab,
                contentType: "application/json",
                dataType: "json",
                data: JSON.stringify({
                    name: name
                })
            }).done(_.bind(function () {
                this.trigger("tabRenamed", tab);
            }, this));
        },
        /**
         * Gets all the fields for specified tab
         * @param {Number} tab - tab id
         * @return {$.Deferred}
         */
        getFieldsForTab: function getFieldsForTab(tab) {
            return safeAjax({
                url: wrmContextPath() + "/rest/api/2/screens/" + this.get("id") + "/tabs/" + tab + "/fields" + this._projectParam()
            });
        },
        /**
         * Deletes tab
         * @param {Number} tab
         * @return {$.Deferred}
         */
        deleteTab: function deleteTab(tab) {
            return safeAjax({
                type: "DELETE",
                url: wrmContextPath() + "/rest/api/2/screens/" + this.get("id") + "/tabs/" + tab
            });
        },

        /**
         * Moves tab position
         * @param {Number} tab - tab id
         * @param {Number} pos - position to move to
         * @return {$.Deferred}
         */
        updateTabPosition: function updateTabPosition(tab, pos) {
            return safeAjax({
                type: "POST",
                url: wrmContextPath() + "/rest/api/2/screens/" + this.get("id") + "/tabs/" + tab + "/move/" + pos
            });
        },

        /**
         * Adds a tab
         * @param {String} name - Name of tab
         * @return {$.Deferred}
         */
        addTab: function addTab(name) {
            //Errors rendered by calling code.
            return jQuery.ajax({
                url: wrmContextPath() + "/rest/api/2/screens/" + this.get("id") + "/tabs/",
                contentType: "application/json",
                dataType: "json",
                data: JSON.stringify({
                    name: name
                }),
                type: "POST"
            }).done(_.bind(function (tab) {
                this.trigger("tabAdded", tab.id);
            }, this));
        },
        /**
         * Gets available fields - List of fields that are not already on the screen
         * @return {$.Deferred}
         */
        getAvailableFields: function getAvailableFields() {
            return safeAjax({
                url: wrmContextPath() + "/rest/api/2/screens/" + this.get("id") + "/availableFields" + this._projectParam()
            });
        }
    });
});