define("jira-project-config/screen-editor", ['jquery', 'jira/analytics', 'jira-project-config/screen-editor/model', 'jira-project-config/screen-editor/view', 'jira/util/logger'], function (jQuery, analytics, ScreenEditorModel, ScreenEditorView, logger) {
    "use strict";

    /**
     * @classdesc The JIRA screen editor.
     *
     * @param {Object} options
     * @param {Element} options.element The element to show the screen editor in.
     * @param {String} [options.analyticsName] The postfix to use when firing analytics events.
     * @param {PCPAnalyticsCommonData} [options.analyticsData=] Properties for analytics events.
     * @param {String} [options.projectKey] The key of the project this screen editor instance is associated with
     * @param {String} [options.screenId] The ID of the screen config to load.
     * @param {Number} [options.selectedTab] The active tab ID.
     * @param {boolean} [options.readOnly] Whether the Screen Editor should be read only or not.
     * @constructor
     */

    return function (options) {
        options.element = jQuery(options.element);

        if (!options.selectedTab && window.sessionStorage) {
            options.selectedTab = window.sessionStorage.getItem("selectedScreenTab");
        }

        this.screenEditorModel = new ScreenEditorModel({
            id: options.screenId,
            projectKey: options.projectKey,
            readOnly: !!options.readOnly
        });

        new ScreenEditorView({
            el: options.element,
            model: this.screenEditorModel
        }).render(options.selectedTab);

        var analyticsData = jQuery.extend({}, options.analyticsData || {}, {
            screenId: this.screenEditorModel.get('id'),
            readOnly: this.screenEditorModel.get('readOnly')
        });

        addScreenEditorAnalytics(options.element, options.analyticsName, analyticsData);

        logger.trace("admin.screeneditor.loaded");
    };

    function addScreenEditorAnalytics(screenEditorEl, suffix, analyticsData) {
        if (!suffix) {
            return;
        }
        if (screenEditorEl.length) {
            analytics.send({ name: "administration.screen.designer.view." + suffix, properties: analyticsData });
            jQuery(screenEditorEl).on({
                "fieldAdded": function fieldAdded() {
                    return analytics.send({ name: "administration.screen.designer.field.added." + suffix });
                },
                "fieldRemoved": function fieldRemoved() {
                    return analytics.send({ name: "administration.screen.designer.field.removed." + suffix });
                },
                "fieldReordered": function fieldReordered() {
                    return analytics.send({ name: "administration.screen.designer.field.reordered." + suffix });
                },
                "tabAdded": function tabAdded() {
                    return analytics.send({ name: "administration.screen.designer.tab.added." + suffix });
                },
                "tabDeleted": function tabDeleted() {
                    return analytics.send({ name: "administration.screen.designer.tab.removed." + suffix });
                },
                "tabRenamed": function tabRenamed() {
                    return analytics.send({ name: "administration.screen.designer.tab.renamed." + suffix });
                },
                "tabReordered": function tabReordered() {
                    return analytics.send({ name: "administration.screen.designer.tab.reordered." + suffix });
                },
                "fieldMovedTab": function fieldMovedTab() {
                    return analytics.send({ name: "administration.screen.designer.field.moved.tab." + suffix });
                }
            });
        }
    }
});