define("jira-project-config/issuetypes/perspectives/workflow/controller", ["require"], function (require) {
    "use strict";

    var _ = require("underscore");
    var HeaderDescriptor = require("jira-project-config/issuetypes/entities/models/header-descriptor");
    var jQuery = require("jquery");
    var Marionette = require("jira-project-config/marionette");
    var Perspective = require("jira-project-config/issuetypes/entities/models/perspective");
    var PluggableActionsView = require("jira-project-config/issuetypes/perspectives/workflow/pluggable-actions/view");
    var PluggableInfoView = require("jira-project-config/issuetypes/perspectives/workflow/workflowinfo/view");
    var SharedByData = require("jira-project-config/issuetypes/entities/models/shared-by-data");
    var WorkflowModel = require("jira-project-config/issuetypes/perspectives/workflow/model");
    var WorkflowView = require("jira-project-config/issuetypes/perspectives/workflow/view");
    var Analytics = require("jira-project-config/issuetypes/perspectives/workflow/analytics");
    var formatter = require("jira/util/formatter");
    var logger = require("jira/util/logger");
    var strings = require("jira/util/strings");
    var analyticsCommonData = require('jira-project-config/global/js/analytics-common-data');

    return Marionette.Controller.extend(
    /** @lends JIRA.ProjectConfig.IssueTypes.Workflow.Controller# */
    {
        /**
         * Initialises the controller.
         *
         * @classdesc The controller responsible for orchestrating the content area when viewing the workflow for an
         *   issue type in a project.
         * @constructs
         * @extends Marionette.Controller
         * @param {Marionette.Wreqr.Commands} options.commands The channel for giving commands. Must support
         *   - `issueTypes:resetPluggableRegions`
         * @param {JIRA.ProjectConfig.IssueTypes.Model} options.model The state of the Issue Types area of the application.
         * @param {Marionette.Region} options.region The region to render content into.
         * @param {Marionette.Wreqr.EventAggregator} options.vent The channel for firing off events and listening for events.
         *   This controller triggers:
         *    - `workflow:editWorkflow` when the edit button is clicked in the actions view
         *    - `workflow:discardComplete` when a workflow has been successfully discarded
         *    - `workflow:publishComplete` when a workflow has been successfully published
         *   Other entities trigger:
         *    - `before:perspectiveRerender` just before a rendering route is executed.
         */
        initialize: function initialize(options) {
            this.commands = options.commands;
            this.pageModel = options.model;
            this.perspectiveModel = new Perspective({
                id: "workflow",
                name: formatter.I18n.getText("admin.issuetypeconfig.perspective.workflow.name")
            });
            this.region = options.region;
            this.sharedByData = new SharedByData();
            this.viewModel = new WorkflowModel();
            this.vent = options.vent;

            // Freshen-up now.
            this.pageModel.get("selectedIssueType") && this._refresh();

            // Stay up-to-date as the parent model changes in the future.
            this.listenTo(this.pageModel, "change:editing change:project change:selectedIssueType", this._refresh);
        },

        /**
         * @returns {JIRA.ProjectConfig.IssueTypes.Entities.Perspective} the model that represents this perspective.
         */
        getPerspectiveModel: function getPerspectiveModel() {
            return this.perspectiveModel;
        },

        /**
         * Sets the region.
         *
         * @param {Marionette.Region} region the region to set
         */
        setRegion: function setRegion(region) {
            this.region = region;
        },

        /**
         * Shows the workflow view in the region, and also makes requests to display the descriptor,
         * to display the shared by data and to display the pluggable actions.
         *
         * @param {Object} [layoutData] the layout data to optionally pass to the Workflow Designer to display.
         */
        show: function show(layoutData) {
            if (this.viewModel.get("editing")) {
                Analytics.triggerEditProject(this._getEventData());
            } else {
                Analytics.triggerViewProject(this._getEventData());
            }

            // Display the pluggable regions
            this.commands.execute("issueTypes:resetPluggableRegions", {
                actions: this._createPluggableActions(),
                info: this._createPluggableInfo(),
                descriptor: this._getDescriptor(),
                sharedBy: this.sharedByData.clone(),
                readOnly: this._getIssueTypeWorkflow()
            });

            // Display the main content
            this._renderWorkflowDesigner(layoutData);
        },

        _getIssueTypeWorkflow: function _getIssueTypeWorkflow() {
            if (this.viewModel.get('issueType')) {
                return this.viewModel.get('issueType').get('workflow');
            }
        },

        /**
         * @returns {JIRA.ProjectConfig.IssueTypes.Entities.HeaderDescriptor} the descriptor that describes this perspective.
         * @private
         */
        _getDescriptor: function _getDescriptor() {
            var issueType = this.viewModel.get("issueType");
            return new HeaderDescriptor({
                title: formatter.I18n.getText("admin.issuetypeconfig.workflow.heading", issueType.get("workflow").get("displayName"))
            });
        },

        /**
         * Get project key.
         *
         * @returns {string|null} project key.
         * @private
         */
        _getProjectKey: function _getProjectKey() {
            if (_.isUndefined(this.pageModel) || _.isNull(this.pageModel) || _.isEmpty(this.pageModel)) {
                return null;
            }
            if (_.isUndefined(this.pageModel.get("project")) || _.isNull(this.pageModel.get("project")) || _.isEmpty(this.pageModel.get("project"))) {
                return null;
            }
            return this.pageModel.get("project").get("key");
        },

        /**
         * @private
         */
        _handlePublishOrDiscardRequest: function _handlePublishOrDiscardRequest(promise, actionsView, completeEvent) {
            var self = this;
            actionsView.disable(true);
            promise.done(function () {
                self.vent.trigger(completeEvent);
            });
            promise.fail(function () {
                actionsView.disable(false);
            });
        },

        /**
         * @param {JIRA.ProjectConfig.IssueTypes.Workflow.PluggableActions.View} actionsView the pluggable actions view
         * @private
         * @deprecated
         */
        _onBeforePerspectiveRerender: function _onBeforePerspectiveRerender(actionsView) {
            actionsView.disable(true);
        },

        /**
         * @param {JIRA.ProjectConfig.IssueTypes.Workflow.PluggableActions.View} actionsView the pluggable actions view
         * @private
         */
        _onDiscardWorkflow: function _onDiscardWorkflow(actionsView) {
            var promise;

            promise = this.designer.discardDraft({
                reloadDesigner: false
            });
            promise.done(function () {
                Analytics.triggerDiscardEditedWorkflow(this._getEventData());
            }.bind(this)).fail(function () {
                Analytics.triggerDiscardEditedWorkflowFail(this._getEventData());
            }.bind(this));

            this._handlePublishOrDiscardRequest(promise, actionsView, "workflow:discardComplete");
        },

        /**
         * @private
         */
        _onEditWorkflow: function _onEditWorkflow() {
            Analytics.triggerEditedWorkflowClicked(this._getEventData());
            this.vent.trigger("workflow:editWorkflow");
        },

        /**
         * @private
         */
        _onExitDraft: function _onExitDraft() {
            Analytics.triggerExitDraftClicked(this._getEventData());
            this.vent.trigger("workflow:exitDraft");
        },

        /**
         * @param {JIRA.ProjectConfig.IssueTypes.Workflow.PluggableActions.View} actionsView the pluggable actions view
         * @private
         */
        _onPublishWorkflow: function _onPublishWorkflow(actionsView) {
            var promise;

            promise = this.designer.publishDraft({
                reloadDesigner: false
            });
            promise.done(function () {
                Analytics.triggerPublishEditedWorkflow(this._getEventData());
            }.bind(this)).fail(function () {
                Analytics.triggerPublishEditedWorkflowFail(this._getEventData());
            }.bind(this));

            this._handlePublishOrDiscardRequest(promise, actionsView, "workflow:publishComplete");
        },

        /**
         * @private
         */
        _refresh: function _refresh() {
            this.viewModel.set({
                editing: this.pageModel.get("editing"),
                issueType: this.pageModel.get("selectedIssueType"),
                project: this.pageModel.get("project")
            });
            this._refreshSharedByData();
        },

        /**
         * @private
         */
        _refreshSharedByData: function _refreshSharedByData() {
            var issueType = this.viewModel.get("issueType");
            var sharedWithIssueTypes = null;
            var sharedWithProjects = null;
            var totalProjectsCount = 0;
            var hiddenProjectsCount = 0;
            var workflow;
            if (issueType) {
                workflow = issueType.get("workflow");
                if (workflow) {
                    sharedWithIssueTypes = workflow.get("sharedWithIssueTypes");
                    sharedWithProjects = workflow.get("sharedWithProjects");
                    totalProjectsCount = workflow.get("totalProjectsCount");
                    hiddenProjectsCount = workflow.get("hiddenProjectsCount");
                }
            }
            this.sharedByData.set({
                projects: sharedWithProjects,
                issueTypes: sharedWithIssueTypes,
                totalProjectsCount: totalProjectsCount,
                hiddenProjectsCount: hiddenProjectsCount,
                issueTypesTitle: formatter.I18n.getText("admin.common.issuetype.used.in.project.list.heading.workflow", sharedWithIssueTypes ? sharedWithIssueTypes.length : 1),
                projectsTitle: formatter.I18n.getText("admin.common.project.used.list.heading.workflow", totalProjectsCount)
            });
        },

        /**
         * Creates the pluggable actions view and attaches the appropriate listeners.
         * @returns {JIRA.ProjectConfig.IssueTypes.Workflow.PluggableActions.View} the view to display in the actions region.
         * @private
         */
        _createPluggableActions: function _createPluggableActions() {
            var pluggableActionsView = new PluggableActionsView({
                model: this.viewModel
            });
            var boundOnBeforePerspectiveRerender = function boundOnBeforePerspectiveRerender() {
                pluggableActionsView.disable(true);
            };

            this.listenTo(pluggableActionsView, "click:edit", _.bind(this._onEditWorkflow, this));
            this.listenTo(pluggableActionsView, "click:exitDraft", _.bind(this._onExitDraft, this));

            this.listenTo(this.vent, "before:perspectiveRerender", boundOnBeforePerspectiveRerender);

            this.listenTo(pluggableActionsView, "close", function () {
                this.stopListening(pluggableActionsView);
                this.stopListening(this.vent, "before:perspectiveRerender", boundOnBeforePerspectiveRerender);
            });

            return pluggableActionsView;
        },

        /**
         * Creates the pluggable info view and attaches the appropriate listeners.
         * @returns {JIRA.ProjectConfig.IssueTypes.Workflow.WorkflowInfo.View} the view to display in the info region.
         * @private
         */
        _createPluggableInfo: function _createPluggableInfo() {
            var pluggableInfoView = new PluggableInfoView({
                model: this.viewModel
            });

            var boundOnBeforePerspectiveRerender = function boundOnBeforePerspectiveRerender() {
                pluggableInfoView.disable(true);
            };

            this.listenTo(pluggableInfoView, "click:discard", _.bind(this._onDiscardWorkflow, this, pluggableInfoView));
            this.listenTo(pluggableInfoView, "click:publish", _.bind(this._onPublishWorkflow, this, pluggableInfoView));

            this.listenTo(this.vent, "before:perspectiveRerender", boundOnBeforePerspectiveRerender);

            this.listenTo(pluggableInfoView, "close", function () {
                this.stopListening(pluggableInfoView);
                this.stopListening(this.vent, "before:perspectiveRerender", boundOnBeforePerspectiveRerender);
            });

            return pluggableInfoView;
        },

        /**
         * Renders the workflow designer with the provided layout data.
         * @param {Object} layoutData the layout data for the workflow designer to display.
         * @private
         */
        _renderWorkflowDesigner: function _renderWorkflowDesigner(layoutData) {
            var _this = this;

            var contentView = new WorkflowView({
                model: this.viewModel
            });
            var workflow = this.viewModel.get("issueType").get("workflow");

            this.region.show(contentView);

            this.designer = new JIRA.WorkflowDesigner.Application({
                element: contentView.ui.workflowDesigner,
                immutable: !this.viewModel.get("editing"),
                layoutData: layoutData,
                workflowId: workflow.get("name"),
                fullScreenButton: false,
                draft: this.viewModel.get("editing"),
                actions: this.viewModel.get("editing"),
                isDraftWithChanges: workflow.get("isDraftWithChanges"),
                analyticsData: this._getEventData()
            });

            this.listenTo(this.designer, "sync", function () {
                var workflow = _this.viewModel.get("issueType").get("workflow");
                if (workflow) {
                    workflow.set("isDraftWithChanges", true);
                }
            });

            this.listenToOnce(contentView, "close", function () {
                this.designer.destroy();
                jQuery(contentView.el).empty();
            });
        },

        /**
         * Prepares event data.
         * It makes use of `pageModel` for obtaining workflow model.
         * `pageModel` may origin either from issuetype or workflow controller.
         *
         * @returns {{isolated: bool, admin: bool, projectId: int, workflowHash: string, workflowState: string, isDraftWithChanges: bool}|{}} or {} in case of error
         * @private
         */
        _getEventData: function _getEventData() {
            var workflow = this.pageModel.get("selectedIssueType") && this.pageModel.get("selectedIssueType").get('workflow');
            if (!workflow || typeof workflow.get !== "function") {
                logger.error("Could not get workflow model");
                return {};
            }

            var isolated = workflow.get("totalProjectsCount") === 1;
            var workflowState = workflow.get("state");
            var workflowHash = strings.hashCode(workflow.get("name"));
            var hasChanges = workflow.get("isDraftWithChanges");

            return _.extend({
                isolated: isolated,
                workflowHash: workflowHash,
                workflowState: workflowState,
                hasChanges: hasChanges
            }, analyticsCommonData);
        }
    });
});