define("jira-project-config/issuetypes/perspectives/fields/controller", ["require"], function (require) {
    "use strict";

    var _ = require('underscore');
    var FieldsView = require('jira-project-config/issuetypes/perspectives/fields/view');
    var FieldsViewModel = require('jira-project-config/issuetypes/perspectives/fields/model');
    var HeaderDescriptor = require('jira-project-config/issuetypes/entities/models/header-descriptor');
    var PluggableInfoView = require('jira-project-config/issuetypes/perspectives/fields/info/view');
    var Marionette = require('jira-project-config/marionette');
    var Perspective = require('jira-project-config/issuetypes/entities/models/perspective');
    var SharedByData = require('jira-project-config/issuetypes/entities/models/shared-by-data');
    var formatter = require('jira/util/formatter');

    return Marionette.Controller.extend(
    /** @lends JIRA.ProjectConfig.IssueTypes.Fields.Controller# */
    {
        /**
         * Initialises the controller.
         *
         * @classdesc The controller responsible for orchestrating the content area of the page when viewing an issue
         *   type's fields for a project.
         * @constructs
         * @extends Marionette.Controller
         * @param {Marionette.Wreqr.Commands} options.commands The channel for giving commands. Must support
         *   - `issueTypes:resetPluggableRegions`
         * @param {JIRA.ProjectConfig.IssueTypes.Model} options.model The state of the page.
         * @param {Marionette.Region} options.region The region to render content into.
         */
        initialize: function initialize(options) {
            this.commands = options.commands;
            this.pageModel = options.model;
            this.region = options.region;
            this.sharedByData = new SharedByData();
            this.viewModel = new FieldsViewModel();
            this.vent = options.vent;

            // Freshen-up now.
            this.pageModel.get("selectedIssueType") && this._refresh();

            this.listenTo(this.pageModel, "change:editing change:project change:selectedIssueType", this._refresh);
        },

        /**
         * @returns {JIRA.ProjectConfig.IssueTypes.Entities.Perspective} the model that represents this perspective.
         */
        getPerspectiveModel: function getPerspectiveModel(screenType) {
            return new Perspective({
                id: "fields",
                screenType: screenType,
                name: this._getPerspectiveName(screenType)
            });
        },

        _getPerspectiveName: function _getPerspectiveName(screenType) {
            switch (screenType) {
                case 'create':
                    return formatter.I18n.getText("admin.issuetypeconfig.perspective.fields.name.create");
                case 'edit':
                    return formatter.I18n.getText("admin.issuetypeconfig.perspective.fields.name.edit");
                case 'view':
                    return formatter.I18n.getText("admin.issuetypeconfig.perspective.fields.name.view");
                default:
                    return formatter.I18n.getText("admin.issuetypeconfig.perspective.fields.name");
            }
        },

        /**
         * Sets the region.
         *
         * @param {Marionette.Region} region the region to set
         */
        setRegion: function setRegion(region) {
            this.region = region;
        },

        /**
         * Shows the fields view in the region and also makes requests to display the descriptor and to
         * display the shared by data.
         */
        show: function show(options) {
            // Display the pluggable regions
            var screenType = options.screenType || 'view';

            this.commands.execute("issueTypes:resetPluggableRegions", {
                descriptor: this._getDescriptor(screenType),
                info: this._createPluggableInfo(screenType),
                readOnly: this._getScreenModel(screenType),
                sharedBy: this.sharedByData.clone()
            });

            // Display the main content
            this.region.show(new FieldsView({
                model: this.viewModel
            }));
        },

        _getScreenModel: function _getScreenModel(screenType) {
            var issueType = this.viewModel.get("issueType");
            return issueType.getScreen(screenType);
        },

        /**
         * @returns {JIRA.ProjectConfig.IssueTypes.Entities.HeaderDescriptor} the descriptor that describes this perspective.
         * @private
         */
        _getDescriptor: function _getDescriptor(screenType) {
            var issueType = this.viewModel.get("issueType");
            var screenName = issueType.get("fields").get("screenName");
            var screen = this._getScreenModel(screenType);
            if (screen) {
                screenName = screen.name;
            }

            return new HeaderDescriptor({
                title: this._getHeaderTitle(screenType, screenName)
            });
        },

        /**
         * Creates the pluggable info view and attaches the appropriate listeners.
         * @returns the view to display in the info region.
         * @private
         */
        _createPluggableInfo: function _createPluggableInfo(screenType) {
            return new PluggableInfoView({
                model: this._getScreenModel(screenType)
            });
        },

        _getHeaderTitle: function _getHeaderTitle(screenType, screenName) {
            switch (screenType) {
                case 'create':
                    return formatter.I18n.getText("admin.issuetypeconfig.fields.heading.create");
                case 'edit':
                    return formatter.I18n.getText("admin.issuetypeconfig.fields.heading.edit");
                case 'view':
                    return formatter.I18n.getText("admin.issuetypeconfig.fields.heading.view");
                default:
                    return formatter.I18n.getText("admin.issuetypeconfig.fields.heading", screenName);
            }
        },

        /**
         * Get project key.
         *
         * @returns {string|null} project key.
         * @private
         */
        _getProjectKey: function _getProjectKey() {
            if (_.isObject(this.pageModel) === false || _.isObject(this.pageModel.get("project")) === false) {
                return null;
            }
            return this.pageModel.get("project").get("key");
        },

        /**
         * Ask for the latest version of the issue types and project.
         * @private
         */
        _refresh: function _refresh() {
            this.viewModel.set({
                issueType: this.pageModel.get("selectedIssueType"),
                project: this.pageModel.get("project")
            });
            this._refreshSharedByData();
            this._refreshScreenMetadata();
        },

        _refreshScreenMetadata: function _refreshScreenMetadata() {
            var issueType = this.viewModel.get("issueType");
            var fields;
            var perspective;
            var currentScreen;

            if (issueType) {
                fields = issueType.get("fields");
                perspective = this.pageModel.get("selectedPerspective");
                if (fields && perspective) {
                    currentScreen = fields.getScreen(perspective.get("screenType"));
                    if (currentScreen) {
                        this.viewModel.set({
                            screenId: currentScreen.get('screenId'),
                            canEdit: currentScreen.get('canEdit')
                        });
                    }
                }
            }
        },

        /**
         * Refreshes the sharedByData property according to the selected issue type's fields.
         *
         * @private
         */
        _refreshSharedByData: function _refreshSharedByData() {
            var issueType = this.viewModel.get("issueType");
            var fields;
            var perspective;
            var currentScreen;
            var sharedWithIssueTypes = null;
            var sharedWithProjects = null;
            var totalProjectsCount = 0;
            var hiddenProjectsCount = 0;
            if (issueType) {
                fields = issueType.get("fields");
                perspective = this.pageModel.get("selectedPerspective");
                if (fields && perspective) {
                    currentScreen = fields.getScreen(perspective.get("screenType"));
                    if (currentScreen) {
                        sharedWithIssueTypes = currentScreen.get("sharedWithIssueTypes");
                        sharedWithProjects = currentScreen.get("sharedWithProjects");
                        totalProjectsCount = currentScreen.get("totalProjectsCount");
                        hiddenProjectsCount = currentScreen.get("hiddenProjectsCount");

                        this.sharedByData.set({
                            projects: sharedWithProjects,
                            issueTypes: sharedWithIssueTypes,
                            totalProjectsCount: totalProjectsCount,
                            hiddenProjectsCount: hiddenProjectsCount,
                            issueTypesTitle: formatter.I18n.getText("admin.common.issuetype.used.in.project.list.heading.fields.set", sharedWithIssueTypes ? sharedWithIssueTypes.length : 1),
                            projectsTitle: formatter.I18n.getText("admin.common.project.used.list.heading.fields.set", totalProjectsCount)
                        });
                    }
                }
            }
        }
    });
});