AJS.test.require(["com.atlassian.jira.jira-project-config-plugin:issuetypes-tab", "com.atlassian.jira.jira-project-config-plugin:issuetypes-test-utils"], function () {
    "use strict";

    var jQuery = require("jquery");
    var _ = require("underscore");
    var formatter = require("jira/util/formatter");
    var Marionette = require("jira-project-config/marionette");
    var Chance = require("jira-project-config/libs/chance");
    var ChanceMixins = require("jira-project-config/issuetypes/entities/test/chance-mixins");
    var IssueTypes = require("jira-project-config/issuetypes/entities/models/issue-types");
    var Projects = require("jira-project-config/issuetypes/entities/models/projects");
    var FieldsController = require("jira-project-config/issuetypes/perspectives/fields/controller");
    var FieldsView = require("jira-project-config/issuetypes/perspectives/fields/view");
    var IssueTypesModel = require("jira-project-config/issuetypes/model");

    module("JIRA.ProjectConfig.IssueTypes.Fields.Controller", {
        setup: function setup() {
            this.$fixture = jQuery("#qunit-fixture");
            this.sandbox = sinon.sandbox.create();

            this.chance = new Chance(1);
            _.extend(this.chance, new ChanceMixins());

            this.project = this.chance.project();
            this.story = this.chance.issueType();

            this.bug = this.chance.issueType().set({ fields: this.chance.fields()
            });

            this.bug.get('fields').getScreen('view').set({
                "sharedWithIssueTypes": new IssueTypes([this.story]),
                "sharedWithProjects": new Projects([this.project]),
                "totalProjectsCount": Math.round(Math.random() * 100),
                "hiddenProjectsCount": Math.round(Math.random() * 100)
            });

            this.model = new IssueTypesModel({
                selectedIssueType: this.bug,
                selectedPerspective: this.bug.get('fields').getScreen('view'),
                project: this.project
            });
            this.region = { show: sinon.stub() };

            this.resetPluggableRegionsSpy = this.sandbox.spy();
            this.commands = new Marionette.Wreqr.Commands();
            this.commands.setHandler("issueTypes:resetPluggableRegions", this.resetPluggableRegionsSpy);
        },

        teardown: function teardown() {
            this.sandbox.restore();
        }
    });

    test("uses the provided region", 3, function () {
        var controller;
        var region = { show: sinon.stub() };

        controller = new FieldsController({
            commands: this.commands,
            model: this.model,
            region: region
        });

        equal(region.show.callCount, 0);
        controller.show({});
        equal(region.show.callCount, 1);
        ok(region.show.firstCall.args[0] instanceof FieldsView);
    });

    test("showing executes issueTypes:resetPluggableRegions with correct information", function () {
        var controller = new FieldsController({
            commands: this.commands,
            model: this.model,
            region: this.region
        });

        var options;

        this.sandbox.spy(formatter, "format");
        controller.show({});

        equal(this.resetPluggableRegionsSpy.callCount, 1, "issueTypes:resetPluggableRegions should have been executed once");
        options = this.resetPluggableRegionsSpy.firstCall.args[0];

        sinon.assert.notCalled(formatter.format);

        deepEqual(options.descriptor.toJSON(), {
            title: "admin.issuetypeconfig.fields.heading.view"
        });

        if (options.sharedBy) {
            deepEqual(options.sharedBy.get("issueTypes").toJSON(), [this.story.toJSON()], "Correct issue type shared");
            deepEqual(options.sharedBy.get("projects").toJSON(), [this.project.toJSON()], "Correct project shared");
            equal(options.sharedBy.get("totalProjectsCount"), this.bug.toJSON().fields.viewScreen.totalProjectsCount, "Correct total projects count");
            equal(options.sharedBy.get("hiddenProjectsCount"), this.bug.toJSON().fields.viewScreen.hiddenProjectsCount, "Correct hidden projects count");
        }
    });

    test("getPerspectiveModel() returns correct information", function () {
        var controller = new FieldsController({
            commands: this.commands,
            model: this.model,
            region: this.region
        });

        ['create', 'edit', 'view'].forEach(function (screenType) {
            var perspectiveModel = controller.getPerspectiveModel(screenType);

            deepEqual(perspectiveModel.toJSON(), {
                id: "fields",
                screenType: screenType,
                name: "admin.issuetypeconfig.perspective.fields.name." + screenType,
                selected: false
            }, "the properties on the perspective model should be correct");
        });

        var perspectiveModel = controller.getPerspectiveModel('other');
        deepEqual(perspectiveModel.toJSON(), {
            id: "fields",
            screenType: 'other',
            name: "admin.issuetypeconfig.perspective.fields.name",
            selected: false
        }, "the properties on the perspective model should be correct");
    });

    test("should return correct header title", function () {
        var controller = new FieldsController({
            commands: this.commands,
            model: this.model,
            region: this.region
        });

        var titleMap = {
            'create': { key: 'admin.issuetypeconfig.fields.heading.create' },
            'edit': { key: 'admin.issuetypeconfig.fields.heading.edit' },
            'view': { key: 'admin.issuetypeconfig.fields.heading.view' },
            'other': { key: 'admin.issuetypeconfig.fields.heading', params: {} }
        };

        _.pairs(titleMap).forEach(function (pair) {
            var screenType = pair[0];
            var title = pair[1];

            this.resetPluggableRegionsSpy.reset();

            controller.show({ screenType: screenType });

            equal(this.resetPluggableRegionsSpy.callCount, 1, "issueTypes:resetPluggableRegions should have been executed once");
            var options = this.resetPluggableRegionsSpy.firstCall.args[0];

            equal(options.descriptor.get('title'), title.key);
        }.bind(this));
    });
});