AJS.test.require(["com.atlassian.jira.jira-project-config-plugin:issuetypes-tab"], function () {
    "use strict";

    var _ = require("underscore");
    var jQuery = require("jquery");
    var SharedBy = require("jira-project-config/issuetypes/header/shared-by/view");
    var IssueType = require('jira-project-config/issuetypes/entities/models/issue-type');
    var IssueTypes = require('jira-project-config/issuetypes/entities/models/issue-types');
    var Projects = require('jira-project-config/issuetypes/entities/models/projects');
    var Project = require('jira-project-config/issuetypes/entities/models/project');
    var SharedByData = require('jira-project-config/issuetypes/entities/models/shared-by-data');

    var SharedByParser = function SharedByParser($link) {
        this.$link = $link;
    };

    _.extend(SharedByParser.prototype, {
        selectors: {
            issueType: ".shared-issuetypes-trigger",
            project: ".shared-projects-trigger"
        },

        hasProjects: function hasProjects() {
            return this._hasLink(this.selectors.project);
        },
        hasIssueTypes: function hasIssueTypes() {
            return this._hasLink(this.selectors.issueType);
        },
        triggerProjects: function triggerProjects() {
            return this._trigger(this.selectors.project);
        },
        triggerIssueTypes: function triggerIssueTypes() {
            return this._trigger(this.selectors.issueType);
        },
        clickAnIssueType: function clickAnIssueType() {
            var issueTypes = this.getIssueTypes();
            if (!issueTypes.length) {
                return false;
            }
            issueTypes.first().click();

            return this;
        },
        getProjectsTitle: function getProjectsTitle() {
            return this._triggerAndGetTitle(this.selectors.project);
        },
        getProjects: function getProjects() {
            return this._triggerAndGetLinks(this.selectors.project);
        },
        getProjectsNames: function getProjectsNames() {
            return this._triggerAndParseList(this.selectors.project);
        },
        getIssueTypesTitle: function getIssueTypesTitle() {
            return this._triggerAndGetTitle(this.selectors.issueType);
        },
        getIssueTypes: function getIssueTypes() {
            return this._triggerAndGetLinks(this.selectors.issueType);
        },
        getIssueTypesNames: function getIssueTypesNames() {
            return this._triggerAndParseList(this.selectors.issueType);
        },
        popupsClosed: function popupsClosed() {
            return !this.$link.find('a.[aria-expanded=true]').length;
        },
        _triggerAndGetTitle: function _triggerAndGetTitle(selector) {
            var $trigger = this._trigger(selector);
            if (!$trigger.length) {
                return "";
            } else {
                var text = this._popup($trigger).find(":header").text();
                return text.replace(/:$/, "");
            }
        },
        _triggerAndParseList: function _triggerAndParseList(selector) {
            var list = this._triggerAndGetLinks(selector).map(function (i, el) {
                return jQuery(el).text();
            });
            return list.length && list.get() || [];
        },
        _triggerAndGetLinks: function _triggerAndGetLinks(selector) {
            var $trigger = this._trigger(selector);
            if (!$trigger.length) {
                return [];
            }

            return this._popup($trigger).find(".shared-item-name");
        },
        _trigger: function _trigger(selector) {
            var $trigger = this.$link.find(selector);
            $trigger.click();
            return $trigger;
        },
        _hasLink: function _hasLink(selector) {
            return !!this.$link.find(selector).length;
        },
        _popup: function _popup($trigger) {
            return this.$link.find($trigger.attr('href'));
        }
    });

    var createIssueTypes = function createIssueTypes() {
        var issueTypes = _.map(arguments, function (value) {
            return new IssueType({
                id: value,
                name: value
            });
        });
        return new IssueTypes(issueTypes);
    };

    /**
     *
     * @returns {Entities.Projects}
     */
    var createProjects = function createProjects() {
        var projects = _.map(arguments, function (name) {
            return new Project({
                name: name
            });
        });
        return new Projects(projects);
    };

    module("JIRA.ProjectConfig.IssueTypes.SharedBy", {
        setup: function setup() {
            this.$fixture = jQuery("#qunit-fixture");
            // AJS.format is used by the Soy templates!
            sinon.stub(AJS, "format", function () {
                return Array.prototype.join.call(arguments, "");
            });
        },
        teardown: function teardown() {
            AJS.format.restore();
        }
    });

    var runTest = function runTest(options) {
        var view = new SharedBy({
            model: new SharedByData({
                projects: createProjects.apply(this, options.projects || []),
                issueTypes: createIssueTypes.apply(this, options.issueTypes || []),
                totalProjectsCount: (options.projects ? options.projects.length : 0) + 1,
                hiddenProjectsCount: options.hiddenProjectsCount || 0,
                projectsTitle: options.projectsTitle || "",
                issueTypesTitle: options.issueTypesTitle || ""
            })
        });
        this.$fixture.html(view.render().$el);

        var parser = new SharedByParser(this.$fixture);

        options.callback(view, parser);
    };

    var assertSharedBy = function assertSharedBy(projects, issueTypes) {
        runTest.call(this, {
            projects: projects,
            issueTypes: issueTypes,
            callback: function callback(view, parser) {
                deepEqual(parser.getProjectsNames(), projects, "Project matches");
                deepEqual(parser.getIssueTypesNames(), issueTypes, "Issue Types matches");
            }
        });
    };

    test("Render projects only", function () {
        assertSharedBy.call(this, ["one", "two"], []);
    });

    test("Render issue types only", function () {
        assertSharedBy.call(this, [], ["one", "two"]);
    });

    test("Render issue types and projects", function () {
        assertSharedBy.call(this, ["project1", "project2"], ["issueType1", "issueType2"]);
    });

    test("Render issue types only if all are in one project", function () {
        runTest.call(this, {
            projects: ["project1"],
            issueTypes: ["type1", "type2"],
            callback: function callback(view, parser) {
                ok(parser.hasIssueTypes(), "Should have issue types");
                ok(!parser.hasProjects(), "Should not have projects");
            }
        });
        runTest.call(this, {
            projects: ["project1", "project2"],
            issueTypes: ["type1", "type2"],
            callback: function callback(view, parser) {
                ok(parser.hasIssueTypes(), "Should have issue types");
                ok(parser.hasProjects(), "Should have projects");
            }
        });
    });

    test("Popups close when view closes", function () {
        runTest.call(this, {
            projects: ["project1", "project2"],
            issueTypes: ["type1"],
            callback: function callback(view, parser) {
                parser.triggerProjects();
                parser.triggerIssueTypes();
                view.close();
                ok(parser.popupsClosed());
            }
        });
    });

    test("Clicking issueType triggers events", function () {
        runTest.call(this, {
            issueTypes: ["type1"],
            callback: function callback(view, parser) {
                var triggered = false;
                view.on("issueType:selected", function () {
                    triggered = true;
                });
                parser.clickAnIssueType();
                ok(triggered, "Clicking issue type triggered 'issueTypeSelected' event.");
            }
        });
    });

    test("Popups have correct title", function () {
        var typesTitle = "issueTypesTitle";
        var projectsTitle = "projectsTitle";
        runTest.call(this, {
            projects: ["project1", "project2"],
            issueTypes: ["type1"],
            issueTypesTitle: typesTitle,
            projectsTitle: projectsTitle,
            callback: function callback(view, parser) {
                strictEqual(parser.getIssueTypesTitle(), typesTitle);
                strictEqual(parser.getProjectsTitle(), projectsTitle);
            }
        });
    });
});