AJS.test.require(["com.atlassian.jira.jira-project-config-plugin:issuetypes-tab"], function () {
    "use strict";

    var _ = require("underscore");
    var jQuery = require("jquery");
    var Backbone = require("jira-project-config/backbone");
    var Perspective = require("jira-project-config/issuetypes/entities/models/perspective");
    var Perspectives = require("jira-project-config/issuetypes/entities/models/perspectives");
    var HeaderPerspectivesView = require("jira-project-config/issuetypes/header/perspectives/view");

    var PerspectivesPageObject = function PerspectivesPageObject($el) {
        this.$el = $el;
    };
    _.extend(PerspectivesPageObject.prototype, {
        getPerspectives: function getPerspectives() {
            return _.map(this.$el.find(".project-issuetype-perspective"), function (el) {
                return new PerspectivePageObject(jQuery(el));
            });
        },

        getPerspectivesText: function getPerspectivesText() {
            return _.map(this.getPerspectives(), function (perspective) {
                return perspective.getText();
            });
        },

        getPerspectiveWithId: function getPerspectiveWithId(id) {
            return _.find(this.getPerspectives(), function (perspective) {
                return "project-issuetypes-perspective-" + id === perspective.getId();
            });
        },

        isHidden: function isHidden() {
            return this.$el.is(":hidden");
        },

        isVisible: function isVisible() {
            return this.$el.is(":visible");
        }
    });

    var PerspectivePageObject = function PerspectivePageObject($el) {
        this.$el = $el;
        this.$a = $el.find('a');
    };
    _.extend(PerspectivePageObject.prototype, {
        click: function click() {
            this.$el.click();
        },

        isSelected: function isSelected() {
            return this.$el.attr("aria-pressed") === "true";
        },

        getId: function getId() {
            return this.$el.attr("id");
        },

        getText: function getText() {
            return this.$el.text();
        }
    });

    module("JIRA.ProjectConfig.IssueTypes.Header.Perspectives.View", {
        setup: function setup() {
            this.perspective1 = new Perspective({
                id: "perspective1",
                name: "Perspective 1"
            });

            this.perspective2 = new Perspective({
                id: "perspective2",
                name: "Perspective 2"
            });

            this.model = new Backbone.Model({
                perspectives: new Perspectives([this.perspective1, this.perspective2]),
                selectedPerspective: this.perspective1
            });

            this.view = new HeaderPerspectivesView({ model: this.model });

            var renderedContent = this.view.render().$el;
            this.perspectivesPageObject = new PerspectivesPageObject(renderedContent);

            jQuery("#qunit-fixture").append(renderedContent);
        }
    });

    test("rendering the view produces non null html", function () {
        ok(this.view.render().$el.html() !== "");
    });

    test("Should mark selected perspective", function () {
        ok(this.perspectivesPageObject.getPerspectiveWithId("perspective1").isSelected());
    });

    test("Perspective selected when model sets selected perspective", function () {
        ok(this.perspectivesPageObject.getPerspectiveWithId("perspective1").isSelected(), "The currently selected perspectives is the first one");
        ok(!this.perspectivesPageObject.getPerspectiveWithId("perspective2").isSelected(), "The currently selected perspectives is not the second one");

        this.model.set("selectedPerspective", this.perspective2);

        ok(!this.perspectivesPageObject.getPerspectiveWithId("perspective1").isSelected(), "The currently selected perspectives is not the first one");
        ok(this.perspectivesPageObject.getPerspectiveWithId("perspective2").isSelected(), "The currently selected perspectives is the second one");
    });

    test("perspective:selected event fired with perspective when perspective clicked", function () {
        var eventSpy = this.sandbox.spy();

        this.view.on("perspective:selected", eventSpy);

        this.perspectivesPageObject.getPerspectiveWithId("perspective2").click();

        equal(eventSpy.callCount, 1, "Event should have been fired.");
        equal(eventSpy.firstCall.args[0], this.perspective2, "The parameter fired with the event should be the newly selected perspective");
    });

    test("Should be visible by default", function () {
        ok(this.perspectivesPageObject.isVisible());
    });

    test("Should hide perspectives when togglePerspectivesVisibility(false) was called", function () {
        this.view.togglePerspectivesVisibility(false);
        ok(this.perspectivesPageObject.isHidden());

        this.view.togglePerspectivesVisibility(false);
        ok(this.perspectivesPageObject.isHidden());
    });

    test("Should show perspectives when togglePerspectivesVisibility(true) was called", function () {
        this.view.togglePerspectivesVisibility(true);
        ok(this.perspectivesPageObject.isVisible());

        this.view.togglePerspectivesVisibility(true);
        ok(this.perspectivesPageObject.isVisible());
    });

    test("Should handle multiple calls of togglePerspectivesVisibility", function () {
        this.view.togglePerspectivesVisibility(false);
        ok(this.perspectivesPageObject.isHidden());

        this.view.togglePerspectivesVisibility(true);
        ok(this.perspectivesPageObject.isVisible());
    });
});