define('jira-project-config/project-config/urls/issue-types/manager', ['underscore'], function (_) {
    "use strict";

    var TEMPLATES = JIRA.Templates.IssueTypeConfig.Urls;

    /**
     * @param {JIRA.ProjectConfig.IssueTypes.Entities.Project|string} options.project the project
     * @param {JIRA.ProjectConfig.IssueTypes.Entities.IssueType|Number} options.issueType the issue type
     * @returns {Object}
     */
    function parseOptions(options) {
        return {
            projectKey: typeof options.project === 'string' ? options.project : options.project.get("key"),
            issueTypeId: typeof options.issueType === 'number' ? options.issueType : options.issueType.get("id"),
            screenType: options.screenType
        };
    }

    /**
     * @class
     * @classdesc An API for retrieving URLs for issue types.
     * @constructor
     */
    function Manager() {}

    _.extend(Manager.prototype, {

        /**
         * Resolves the base URL that corresponds to the provided project key and issue type.
         *
         * @param {JIRA.ProjectConfig.IssueTypes.Entities.Project|string} options.project the project
         * @param {JIRA.ProjectConfig.IssueTypes.Entities.IssueType|Number} options.issueType the issue type
         * @returns {String} the base URL that represents this project key and issue type
         */
        base: function base(options) {
            return TEMPLATES.base(parseOptions(options));
        },

        /**
         * Resolves the View Workflow URL that corresponds to the provided project key and issue type.
         *
         * @param {JIRA.ProjectConfig.IssueTypes.Entities.Project|string} options.project the project
         * @param {JIRA.ProjectConfig.IssueTypes.Entities.IssueType|Number} options.issueType the issue type
         * @returns {String} the View Workflow URL that represents this project key and issue type
         */
        viewWorkflow: function viewWorkflow(options) {
            return TEMPLATES.viewWorkflow(parseOptions(options));
        },

        /**
         * Resolves the Edit Workflow URL that corresponds to the provided project key and issue type.
         *
         * @param {JIRA.ProjectConfig.IssueTypes.Entities.Project|string} options.project the project
         * @param {JIRA.ProjectConfig.IssueTypes.Entities.IssueType|Number} options.issueType the issue type
         * @returns {String} the Edit Workflow URL that represents this project key and issue type
         */
        editWorkflow: function editWorkflow(options) {
            return TEMPLATES.editWorkflow(parseOptions(options));
        },

        /**
         * Resolves the View Fields URL that corresponds to the provided project key and issue type.
         *
         * @param {JIRA.ProjectConfig.IssueTypes.Entities.Project|string} options.project the project
         * @param {JIRA.ProjectConfig.IssueTypes.Entities.IssueType|Number} options.issueType the issue type
         * @param {string} options.screenType
         * @returns {String} the View Fields URL that represents this project key and issue type
         */
        viewFields: function viewFields(options) {
            return TEMPLATES.viewFields(parseOptions(options));
        },

        /**
         * Resolves the REST API URL for the Fields of an issue type in a particular project.
         *
         * @param {JIRA.ProjectConfig.IssueTypes.Entities.Project|string} options.project the project
         * @param {JIRA.ProjectConfig.IssueTypes.Entities.IssueType|Number} options.issueType the issue type
         * @returns {String} A URL.
         */
        projectIssueTypeFields: function projectIssueTypeFields(options) {
            return TEMPLATES.projectIssueTypeFields(parseOptions(options));
        },

        /**
         * Resolves the REST API URL for the workflow of an issue type in a particular project.
         *
         * @param {JIRA.ProjectConfig.IssueTypes.Entities.Project|string} options.project the project
         * @param {JIRA.ProjectConfig.IssueTypes.Entities.IssueType|Number} options.issueType the issue type
         * @returns {String} A URL.
         */
        projectIssueTypeWorkflow: function projectIssueTypeWorkflow(options) {
            return TEMPLATES.projectIssueTypeWorkflow(parseOptions(options));
        }
    });

    return Manager;
});

AJS.namespace('JIRA.ProjectConfig.Urls.IssueTypes.Manager', null, require('jira-project-config/project-config/urls/issue-types/manager'));