/**
 * @module quick-edit/util/jira
 */
define('quick-edit/util/jira', [
    'quick-edit/templates/issue',
    'jira/util/data/meta',
    'aui/message',
    'jquery'
], function(
    IssueTemplates,
    Meta,
    AuiMessage,
    jQuery
){
    'use strict';

    /**
     * If the child element is within an aui tab, will make that tab active
     * @param {jQuery} child
     */
    var activateTabWithChild = function (child) {
        var tabId,
            $tabContainer,
            $tabTrigger,
            $tab = child.closest(".tabs-pane");

        if ($tab.length) {
            $tabContainer = $tab.closest(".aui-tabs");
            tabId = $tab.attr("id");
            $tabTrigger = $tabContainer.find("a[href='#" + tabId + "']");
            $tabTrigger.click();
        }
    };

    /**
     * Adds an error message to the form
     *
     * @param {jQuery} form
     * @param {String} error
     */
    var applyErrorMessageToForm = function (form, error) {
        return applyMessageToForm("error", form, AJS.escapeHtml(error), false);
    };

    /**
     * Adds an error message to the form without escaping html
     *
     * @param {jQuery} form
     * @param {String} error
     */
    var applyErrorMessageWithHTMLToForm = function (form, error) {
        return applyMessageToForm("error", form, error, false);
    };

    /**
     * Appends inline errors to form, focusing the first field with error
     *
     * @param {jQuery} form
     * @param {Object<fieldName>:<errorMessage>} errors
     */
    var applyErrorsToForm = function (form, errors) {
        var $focusField,
            undealtErrors = [];

        jQuery.each(errors, function (name, message) {
            var $group,
                $error,
                $field = form.find(":input[name='" + name + "']").last();

            if ($field.length === 1) {
                if (!$focusField) {
                    $focusField = $field; // store first field with error so we can focus it at the end
                }
                $error = jQuery("<div class='error' />").attr("data-field", name).text(message);
                $group = $field.closest(".field-group, .group");
                $group.find(".error").remove(); // remove any pre-existing errors
                $group.append($error);
            } else if ($field.length === 0) {
                undealtErrors.push(message);
            }
        });

        if (undealtErrors.length > 0) {
            applyErrorMessageWithHTMLToForm(form, IssueTemplates.undealtErrors({errors: undealtErrors}));
        }

        activateTabWithChild(form.find(".error:first"));

        if ($focusField) {
            $focusField.focus();
        }
    };

    /**
     * Prepends error message (aui style) to form body
     *
     * @param {jQuery} form
     * @param {String} error
     */
    var applyMessageToForm = function (type, form, message, dismissable) {
        var $errorCtx = form.find(".aui-message-context");

        if (!$errorCtx.length) {
            $errorCtx = jQuery("<div class='aui-message-context' />");
            $errorCtx.prependTo(form.find(".form-body"));
        }

        $errorCtx.empty();

        AuiMessage[type]($errorCtx, {
            shadowed: false,
            closeable: dismissable,
            body: message
        });
    };

    /**
     * Adds a success message to the form
     * @param {jQuery} form
     * @param {String} message
     */
    var applySuccessMessageToForm = function (form, message) {
        return applyMessageToForm("success", form, message, true);
    };

    /**
     * Extracts script tags from html fragments.
     *
     * jQuery will remove any script tags in the supplied html and
     * append them separately to the result (var $html). It does this to
     * allow ajax responses to contain script elements which are evaluated
     * when appended.
     *
     * Since we want to run the scripts in our own time we'll strip them
     * out and return them in their own object.
     *
     * @param html
     */
    var extractScripts = function (html) {
        var $html = jQuery(html);

        return {
            html: $html.not("script"),
            scripts: $html.filter("script")
        }
    };

    /**
     * Wraps a property as a function if it is not already one
     *
     * @param property
     * @return function
     */
    var makePropertyFunc = function (property) {
        if (jQuery.isFunction(property)) {
            return property;
        } else {
            return function () {
                return property;
            }
        }
    };

    var getModifierKey = function () {
        Meta.get("keyboard-accesskey-modifier");
    };

    return {
        activateTabWithChild: activateTabWithChild,
        applyErrorMessageToForm: applyErrorMessageToForm,
        applyErrorsToForm: applyErrorsToForm,
        applyMessageToForm: applyMessageToForm,
        applySuccessMessageToForm: applySuccessMessageToForm,
        getModifierKey: getModifierKey,
        extractScripts: extractScripts,
        makePropertyFunc: makePropertyFunc
    };
});

/**
 * @deprecated JIRA.activateTabWithChild
 */
AJS.namespace('JIRA.activateTabWithChild', null, require('quick-edit/util/jira').activateTabWithChild);

/**
 * @deprecated JIRA.applyErrorMessageToForm
 */
AJS.namespace('JIRA.applyErrorMessageToForm', null, require('quick-edit/util/jira').applyErrorMessageToForm);

/**
 * @deprecated JIRA.applyErrorsToForm
 */
AJS.namespace('JIRA.applyErrorsToForm', null, require('quick-edit/util/jira').applyErrorsToForm);

/**
 * @deprecated JIRA.applyMessageToForm
 */
AJS.namespace('JIRA.applyMessageToForm', null, require('quick-edit/util/jira').applyMessageToForm);

/**
 * @deprecated JIRA.applySuccessMessageToFor
 */
AJS.namespace('JIRA.applySuccessMessageToForm', null, require('quick-edit/util/jira').applySuccessMessageToForm);

/**
 * @deprecated JIRA.extractScripts
 */
AJS.namespace('JIRA.extractScripts', null, require('quick-edit/util/jira').extractScripts);

/**
 * @deprecated JIRA.makePropertyFunc
 */
AJS.namespace('JIRA.makePropertyFunc', null, require('quick-edit/util/jira').makePropertyFunc);
