/**
 * @module quick-edit/form/model/create-issue
 */
define('quick-edit/form/model/create-issue', [
    'quick-edit/form/model',
    'wrm/context-path',
    'underscore',
    'jquery'
], function(
    FormModel,
    contextPath,
    _,
    jQuery
){
    'use strict';

    /**
     * A special Model for quick create. Has special handling for setup fields, project an issue type
     *
     * @class CreateIssueFormModel
     * @extends FormModel
     */
    return FormModel.extend({
        DEFAULT_NON_RETAINED_FIELDS : ["summary", "description", "timetracking", "timetracking_originalestimate",
            "timetracking_remainingestimate", "worklog", "worklog_startDate", "worklog_timeLogged", "worklog_newEstimate",
            "worklog_adjustmentAmount", "attachment"],

        CUSTOMFIELD_PATTERN: /^(customfield_(\d+))(?:|\:(?:\d+))$/,
        TIMETRACKING_PATTERN: /^timetracking_/,

        /**
         * @constructor
         * @param {object} options
         * ... {String} issueType - The initially selected issue type.
         * ... {String} projectId - The initially selected project id.
         * ... {String} parentIssueId - The parent issue id.
         * ... {Array} nonRetainedFields - An array on non retained fields between quick creates. this will be extended with the defaults.
         */
        init: function (options) {
            options = options || {};

            this.userFieldsResource = contextPath() + "/rest/quickedit/1.0/userpreferences/create";
            this._hasRetainFeature = true;
            this._hasVisibilityFeature = true;
            this.retainedFields = [];
            this.dirtyFields = [];
            // initIssueType and initProjectId allow you to override the default project and issue type.
            this.initIssueType = options.issueType;
            this.initProjectId = options.projectId;
            this.parentIssueId = options.parentIssueId;

            this.nonRetainedFields = this.DEFAULT_NON_RETAINED_FIELDS;
            if (options.nonRetainedFields) {
                this.nonRetainedFields = _.union(options.nonRetainedFields, this.DEFAULT_NON_RETAINED_FIELDS);
            }
        },

        /**
         * Gets parent issue id, if there is one
         * @return {String}
         */
        getParentIssueId: function () {
            return this.parentIssueId;
        },

        /**
         * Checks to see if we are in multi create mode
         *
         * @return {Boolean}
         */
        isInMultipleMode: function () {
            return this.multipleMode;
        },

        /**
         * Sets if we are creating more than one issue at a time
         *
         * @param {Boolean} state
         */
        setIsMultipleMode: function (state) {
            this.multipleMode = state;
        },

        /**
         * Gets active fields. For Create Issue, this means all required fields also
         */
        getActiveFieldIds: function () {
            var instance = this,
                deferred = jQuery.Deferred(),
                activeFieldIds = [];

            this.getUserFields().done(function (userfields) {
                jQuery.each(userfields, function (i, fieldId) {
                    activeFieldIds.push(fieldId);
                });

                instance.getRequiredFields().done(function (requiredFields) {
                    jQuery.each(requiredFields, function (i, requiredField) {
                        if (jQuery.inArray(requiredField.id, activeFieldIds) === -1) {
                            activeFieldIds.push(requiredField.id);
                        }
                    });
                    deferred.resolve(activeFieldIds);
                });
            });

            return deferred.promise();
        },

        /**
         * Specifies on a per field basis if it's visibility can be configured. Can it be added and removed from the form.
         * In the case of quick create, required fields can not be removed.
         *
         * @param descriptor - field descriptor
         * @return {Boolean}
         */
        hasVisibilityFeature: function (descriptor) {
            return !descriptor.required;
        },

        /**
         * Gets fields resource url, adding initIssueType & initProjectId to the request if available.
         *
         * initIssueType and initProjectId allow you to override the default project and issue type.
         *
         * @return {String}
         */
        getFieldsResource: function () {
            var fieldsResource =  contextPath() + "/secure/QuickCreateIssue!default.jspa?decorator=none";

            if (this.parentIssueId) {
                return fieldsResource + "&parentIssueId=" + this.parentIssueId;
            }

            if (this.initIssueType) {
                fieldsResource = fieldsResource + "&issuetype=" + this.initIssueType;
                delete this.initIssueType;
            }

            if (this.initProjectId) {
                fieldsResource = fieldsResource + "&pid=" + this.initProjectId;
                delete this.initProjectId;
            }

            return fieldsResource;
        },

        /**
         * Sets field values to retain.
         * Note, you can only force retention of fields that are not in the non retained fields list.
         *
         * @param id
         * @param force - always retain.
         */
        addFieldToRetainValue: function (id, force) {
            var isRetainedField = !_.contains(this.nonRetainedFields, id);
            var isNotFileAttachmentField = id.match(/^filetoconvert/) !== null; // don't track these fields

            if (isRetainedField || isNotFileAttachmentField || force) {
                this._super(id);
            }
        },

        /**
         * Gets required fields
         */
        getRequiredFields: function () {
            var deferred = jQuery.Deferred(),
                requiredFields = [];

            this.getFields().done(function (fields) {
                jQuery.each(fields, function (i, field) {
                    if (field.required) {
                        requiredFields.push(field);
                    }
                });
                deferred.resolve(requiredFields);
            });

            return deferred.promise();
        },


        /**
         * Gets fields that can be configured. Configured meaning, fields that can have features such as pinned values and
         * visibility toggling applied to them
         *
         * @return {jQuery.Promise}
         */
        getConfigurableFields: function () {
            var deferred = jQuery.Deferred(),
                issueFields = [];

            this.getFields().done(function (fields) {
                jQuery.each(fields, function (i, field) {
                    if (field.id !== "project" && field.id !== "issuetype") {
                        issueFields.push(field);
                    }
                });

                deferred.resolve(issueFields);
            });

            return deferred.promise();
        },

        /**
         * Gets project and issue type field. Used to get the correct fields for creating an issue.
         */
        getIssueSetupFields: function () {
            var deferred = jQuery.Deferred(),
                issueSetupFields = [];

            this.getFields().done(function (fields) {
                jQuery.each(fields, function (i, field) {
                    if (field.id === "project" || field.id === "issuetype") {
                        issueSetupFields.push(field);
                    }
                });
                deferred.resolve(issueSetupFields);
            });

            return deferred.promise();
        },

        /**
         * Mark given field as dirty, telling us that it's value was provided by user
         *
         * @param fieldId
         */
        markFieldAsDirty: function(fieldId) {
            fieldId = this.normalizeCustomFieldId(fieldId);

            if(!this.fieldIsDirty(fieldId)) {
                this.dirtyFields.push(fieldId);
            }
        },

        /**
         * Check whether field value was provided by user
         *
         * @param fieldId
         * @returns {Boolean}
         */
        fieldIsDirty: function(fieldId) {
            return _.contains(this.dirtyFields, fieldId);
        },

        /**
         * For given field id it will return true if it is coming from a custom field
         * @param fieldId
         * @returns {Boolean}
         */
        isCustomFieldId: function(fieldId) {
            return fieldId && !!fieldId.match(this.CUSTOMFIELD_PATTERN);
        },

        isTimetrackingFieldId: function(fieldId) {
            return fieldId && !!fieldId.match(this.TIMETRACKING_PATTERN);
        },

        /**
         * For custom fields with multiple inputs it will return id that does not contain any suffix (i.e. ":1" for cascading select)
         * @param fieldId
         * @returns {String}
         */
        normalizeCustomFieldId: function(fieldId) {
            if(this.isCustomFieldId(fieldId)) {
                return fieldId.match(this.CUSTOMFIELD_PATTERN)[1];
            } else {
                return fieldId;
            }
        },

        normalizeFieldParam: function(paramName, paramValue) {
            if(paramName == 'filetoconvert') {
                return paramName + '-' + paramValue;
            } else if(this.isCustomFieldId(paramName)) {
                return this.normalizeCustomFieldId(paramName);
            } else if(this.isTimetrackingFieldId(paramName)) {
                return 'timetracking';
            } else {
                return paramName;
            }
        }

    });
});

/**
 * @deprecated JIRA.Forms.CreateIssueModel
 */
AJS.namespace('JIRA.Forms.CreateIssueModel', null, require('quick-edit/form/model/create-issue'));
