define('jira/project-templates/project-key-generator', [
    'jquery',
    'underscore'
], function(
    $,
    _
){
    var ProjectKeyGenerator = {};
    ProjectKeyGenerator.IGNORED_WORDS = ["THE", "A", "AN", "AS", "AND", "OF", "OR"];

    // The (non-ascii) characters used as keys will be replaced with their (ascii) value.
    ProjectKeyGenerator.CHARACTER_MAP = {};
    ProjectKeyGenerator.CHARACTER_MAP[199] = "C"; // ?
    ProjectKeyGenerator.CHARACTER_MAP[231] = "c"; // ?
    ProjectKeyGenerator.CHARACTER_MAP[252] = "u"; // ?
    ProjectKeyGenerator.CHARACTER_MAP[251] = "u"; // ?
    ProjectKeyGenerator.CHARACTER_MAP[250] = "u"; // ?
    ProjectKeyGenerator.CHARACTER_MAP[249] = "u"; // ?
    ProjectKeyGenerator.CHARACTER_MAP[233] = "e"; // ?
    ProjectKeyGenerator.CHARACTER_MAP[234] = "e"; // ?
    ProjectKeyGenerator.CHARACTER_MAP[235] = "e"; // ?
    ProjectKeyGenerator.CHARACTER_MAP[232] = "e"; // ?
    ProjectKeyGenerator.CHARACTER_MAP[226] = "a"; // ?
    ProjectKeyGenerator.CHARACTER_MAP[228] = "a"; // ?
    ProjectKeyGenerator.CHARACTER_MAP[224] = "a"; // ?
    ProjectKeyGenerator.CHARACTER_MAP[229] = "a"; // ?
    ProjectKeyGenerator.CHARACTER_MAP[225] = "a"; // ?
    ProjectKeyGenerator.CHARACTER_MAP[239] = "i"; // ?
    ProjectKeyGenerator.CHARACTER_MAP[238] = "i"; // ?
    ProjectKeyGenerator.CHARACTER_MAP[236] = "i"; // ?
    ProjectKeyGenerator.CHARACTER_MAP[237] = "i"; // ?
    ProjectKeyGenerator.CHARACTER_MAP[196] = "A"; // ?
    ProjectKeyGenerator.CHARACTER_MAP[197] = "A"; // ?
    ProjectKeyGenerator.CHARACTER_MAP[201] = "E"; // ?
    ProjectKeyGenerator.CHARACTER_MAP[230] = "ae"; // ?
    ProjectKeyGenerator.CHARACTER_MAP[198] = "Ae"; // ®
    ProjectKeyGenerator.CHARACTER_MAP[244] = "o"; // ?
    ProjectKeyGenerator.CHARACTER_MAP[246] = "o"; // ?
    ProjectKeyGenerator.CHARACTER_MAP[242] = "o"; // ?
    ProjectKeyGenerator.CHARACTER_MAP[243] = "o"; // ?
    ProjectKeyGenerator.CHARACTER_MAP[220] = "U"; // ?
    ProjectKeyGenerator.CHARACTER_MAP[255] = "Y"; // Ø
    ProjectKeyGenerator.CHARACTER_MAP[214] = "O"; // ?
    ProjectKeyGenerator.CHARACTER_MAP[241] = "n"; // ?
    ProjectKeyGenerator.CHARACTER_MAP[209] = "N"; // ?

    ProjectKeyGenerator.desiredKeyLength = 4;
    ProjectKeyGenerator.maxKeyLength = 10;

    ProjectKeyGenerator.getTotalLength = function getTotalLength(words) {
        return words.join("").length;
    };

    ProjectKeyGenerator.removeIgnoredWords = function removeIgnoredWords(words) {
        return _.reject(words, function(word) {
            return $.inArray(word, ProjectKeyGenerator.IGNORED_WORDS) !== -1;
        });
    };

    ProjectKeyGenerator.createAcronym = function createAcronym(words) {
        var result = "";
        $.each(words, function(i, word) {
            result += word.charAt(0);
        });
        return result;
    };

    ProjectKeyGenerator.getFirstSyllable = function getFirstSyllable(word) {
        // Best guess at getting the first syllable
        // Returns the substring up to and including the first consonant to appear after a vowel
        var pastVowel = false;
        var i;
        for (i = 0; i < word.length; i++) {
            if (ProjectKeyGenerator.isVowelOrY(word[i])) {
                pastVowel = true;
            } else {
                if (pastVowel) {
                    return word.substring(0, i + 1);
                }
            }
        }
        return word;
    };

    ProjectKeyGenerator.isVowelOrY = function isVowelOrY(c) {
        return c && c.length === 1 && c.search("[AEIOUY]") !== -1;
    };

    ProjectKeyGenerator.init = function init(desiredKeyLength, maxKeyLength) {
        ProjectKeyGenerator.desiredKeyLength = desiredKeyLength;
        ProjectKeyGenerator.maxKeyLength = maxKeyLength;
    };

    ProjectKeyGenerator.generate = function generate(name) {
        name = $.trim(name);
        if (!name) {
            return "";
        }

        // Brute-force chunk-by-chunk substitution and filtering.
        var filtered = [];
        for(var i=0, ii=name.length; i<ii; i++) {
            var sub = ProjectKeyGenerator.CHARACTER_MAP[name.charCodeAt(i)];
            filtered.push(sub ? sub : name[i]);
        }
        name = filtered.join('');

        // Split into words
        var words = [];
        $.each(name.split(/\s+/), function(i, word) {
            if (word) {
                // Remove whitespace and punctuation characters (i.e. anything not A-Z)
                word = word.replace(/[^a-zA-Z]/g, "");
                // uppercase the word (NOTE: JavaScript attempts to convert characters like § in to SS)
                word = word.toUpperCase();
                // add the word, should it be worthy.
                word.length && words.push(word);
            }
        });

        // Remove ignored words
        if (ProjectKeyGenerator.desiredKeyLength && ProjectKeyGenerator.getTotalLength(words) > ProjectKeyGenerator.desiredKeyLength) {
            words = ProjectKeyGenerator.removeIgnoredWords(words);
        }

        var key;

        if (words.length == 0) {
            // No words were worthy!
            key = "";
        } else if (words.length == 1) {
            // If we have one word, and it is longer than a desired key, get the first syllable
            var word = words[0];
            if (ProjectKeyGenerator.desiredKeyLength && word.length > ProjectKeyGenerator.desiredKeyLength) {
                key = ProjectKeyGenerator.getFirstSyllable(word);
            } else {
                // The word is short enough to use as a key
                key = word;
            }
        } else {
            // If we have more than one word, just take the first letter from each
            key = ProjectKeyGenerator.createAcronym(words);
        }

        // Limit the length of the key
        if (ProjectKeyGenerator.maxKeyLength && key.length > ProjectKeyGenerator.maxKeyLength) {
            key = key.substr(0, ProjectKeyGenerator.maxKeyLength);
        }

        return key;
    };
    return ProjectKeyGenerator;
});
AJS.namespace('JPT.ProjectKeyGenerator', null, require('jira/project-templates/project-key-generator'));