AJS.test.require([
    "com.atlassian.jira.jira-issue-nav-plugin:testutils",
    "com.atlassian.jira.jira-issue-nav-plugin:issuenav"
], function(){
    "use strict";

    module("JIRA.Issues.Utils.debounce", {
        setup: function () {
            this.myView = {
                callCount: 0,
                render: function () {
                    this.callCount++;
                }
            };
        }
    });

    test("first call executes straight away", function () {
        JIRA.Issues.Utils.debounce(this.myView, "render");
        equal(this.myView.callCount, 1, "Expected render to be called straight away");
    });

    test("debounce defers execution until input stops", function () {
        var clock = sinon.useFakeTimers();
        JIRA.Issues.Utils.debounce(this.myView, "render");

        function f() { this.myView.render();}
        for (var i = 0; i < 100; i++) {
            window.setTimeout(_.bind(f, this), 10);
        }
        clock.tick(1000);

        equal(this.myView.callCount, 2, "Expected render to be called straight away");
        clock.restore();
    });

    test("debounce stops debouncing after 500ms of last execution", function () {
        var clock = sinon.useFakeTimers();
        JIRA.Issues.Utils.debounce(this.myView, "render");

        function f() { this.myView.render();}
        for (var i = 0; i < 100; i++) {
            window.setTimeout(_.bind(f, this), 10);
        }
        clock.tick(1500);

        this.myView.render();

        equal(this.myView.callCount, 3, "Expected render to be called straight away");
        clock.restore();
    });


    test("debounce does not apply twice if called consecutively", function () {
        var clock = sinon.useFakeTimers();

        JIRA.Issues.Utils.debounce(this.myView, "render");
        JIRA.Issues.Utils.debounce(this.myView, "render");
        JIRA.Issues.Utils.debounce(this.myView, "render");
        JIRA.Issues.Utils.debounce(this.myView, "render");
        JIRA.Issues.Utils.debounce(this.myView, "render");
        JIRA.Issues.Utils.debounce(this.myView, "render");
        JIRA.Issues.Utils.debounce(this.myView, "render");
        JIRA.Issues.Utils.debounce(this.myView, "render");
        JIRA.Issues.Utils.debounce(this.myView, "render");
        JIRA.Issues.Utils.debounce(this.myView, "render");

        function f() {this.myView.render();}
        for (var i = 0; i < 100; i++) {
            window.setTimeout(_.bind(f, this), 10);
        }
        clock.tick(1000);

        equal(this.myView.callCount, 2, "Expected delay not to accumulate");
        clock.restore();
    });

    test("patch restores non-existent attributes", function () {
        var obj = {};
        var restore;

        restore = JIRA.Issues.Utils.patch(obj, "foo", "bar");
        restore();

        equal("foo" in obj, false, "key 'foo' shouldn't exist");
    });

    test("patch replaces attribute", function () {
        var obj = {};

        JIRA.Issues.Utils.patch(obj, "foo", "bar");
        equal(obj.foo, "bar");
    });

    test("patch restores original attributes", function () {
        var obj = {foo: "baz"};
        var restore;

        restore = JIRA.Issues.Utils.patch(obj, "foo", "bar");
        equal(obj.foo, "bar");

        restore();
        equal(obj.foo, "baz");
    });

    test("Check all specified cases for empty string", function () {
        var specMap = new Map();
        specMap.set(null, true);
        specMap.set(undefined, true);
        specMap.set("    ", true);
        specMap.set("", true);
        specMap.set("test", false);
        specMap.set("   t", false);

        specMap.forEach(function (expectedResult, valueUnderTest) {
            var computedValue = JIRA.Issues.Utils.isEmptyOrWhitespace(valueUnderTest);
            equal(expectedResult, computedValue, "Is tested string: \"" + valueUnderTest + "\" empty? "
                + expectedResult + ". Method returned: " + computedValue);
        });
    });
});
