define('jira/issues/search/issue/full-screen-issue', ['require'], function (require) {
    "use strict";

    var trace = require('jira/issues/libs/trace');
    var BaseView = require('jira/issues/baseview');
    var Application = require('jira/issues/application');
    var Events = require('jira/util/events');
    var Types = require('jira/util/events/types');
    var Reasons = require('jira/util/events/reasons');
    var _ = require('underscore');
    var jQuery = require('jquery');
    var notification = require('jira/issues/notifications');

    return BaseView.extend({

        namedEvents: ["issueHidden"],

        initialize: function(options) {
            _.extend(this, options);
            _.bindAll(this,
                "onLoadComplete",
                "showError");

            if (this.issueContainer.hasClass("hidden")) {
                this.issueContainer.detach();
                this.issueContainer.removeClass("hidden");
            }

            if (this.searchContainer.hasClass("hidden")) {
                this.searchContainer.detach();
                this.searchContainer.removeClass("hidden");
            }
        },

        bindSearchService: function(searchService) {
            this.listenTo(searchService, {
                "issueUpdated": function(issueId, entity, reason) {
                    if (reason.action !== "inlineEdit" && reason.action !== "rowUpdate") {
                        return this.updateIssue(reason);
                    }
                },
                "selectedIssueChanged": function(selectedIssue) {
                    if (selectedIssue.hasIssue()) {
                        Application.execute("issueEditor:abortPending");
                        //TODO Why do we need to debounce this?
                        //JIRA.Issues.Utils.debounce(this, "_loadIssue", issue);
                        this.show({
                            id: selectedIssue.getId(),
                            key: selectedIssue.getKey(),
                            pager: searchService.getPager()
                        });
                        this.updatePager(searchService.getPager());
                    } else {
                        if (this.isVisible()) {
                            this.hide();
                            trace("jira.returned.to.search");
                            trace("jira.psycho.returned.to.search");
                            Application.execute("analytics:trigger", 'kickass.returntosearch');
                        }
                        //TODO Why the full $navigatorContent is marked as pending?
                        //this.$navigatorContent.removeClass("pending");
                    }
                }
            });
        },

        deactivate: function() {
            this.stopListening();
            if (this.active) {
                if (this.stalker) {
                    this.stalker.unstalk();
                }
                Application.off("issueEditor:loadComplete", this.onLoadComplete);
                Application.off("issueEditor:loadError", this.showError);
                Application.execute("issueEditor:abortPending");
                this.active = false;
            }
        },

        activate: function() {
            if (!this.active) {
                Application.on("issueEditor:loadComplete", this.onLoadComplete);
                Application.on("issueEditor:loadError", this.showError);
                this.active = true;
            }
        },

        onLoadComplete: function(model, props) {
            this.stalker = jQuery(".js-stalker", this.issueContainer).stalker();
            this._makeIssueVisible();
            if (props.isNewIssue) {
                this._scrollToTop();
            }
        },

        _makeIssueVisible: function() {
            Application.execute("issueEditor:beforeShow");
            this._setBodyClasses({
                error: false,
                issue: true,
                search: true
            });

            if (!this.isVisible()) {
                this.issueContainer.insertBefore(this.searchContainer);
                this.searchContainer.detach();

                trace(
                    "jira.psycho.issue.refreshed",
                    {id: Application.request("issueEditor:getIssueId")}
                );
                Events.trigger(Types.NEW_CONTENT_ADDED, [this.issueContainer, Reasons.pageLoad]);
            }
        },

        /**
         * Refresh the visible issue in response to an issue update.
         *
         * @param {object} issueUpdate An issue update object (see <tt>JIRA.Issues.Utils.getUpdateCommandForDialog</tt>).
         * @return {jQuery.Deferred} A deferred that is resolved after the issue has been refreshed.
         */
        updateIssue: function(issueUpdate) {
            var deferred;
            var isVisibleIssue;

            isVisibleIssue = issueUpdate.key === Application.request("issueEditor:getIssueKey");

            if (this.isVisible()) {
                deferred = Application.request("issueEditor:refreshIssue", issueUpdate);
                deferred.done(function() {
                    if (!isVisibleIssue && issueUpdate.message) {
                        notification(issueUpdate.message, issueUpdate.key);
                    }
                });
            } else {
                deferred = new jQuery.Deferred().resolve().promise();
            }

            return deferred;
        },

        /**
         * Show error message for loading issue
         */
        showError: function() {
            this._setBodyClasses({
                error: true,
                issue: false,
                search: false
            });
        },

        /**
         * @return {boolean} whether an issue is visible.
         */
        isVisible: function() {
            return this.issueContainer.closest("body").length > 0;
        },

        /**
         * Scroll to the top of the window.
         *
         * @private
         */
        _scrollToTop: function() {
            jQuery(window).scrollTop(0);
        },

        _setBodyClasses: function(options) {
            jQuery("body")
                .toggleClass("page-type-message", options.error)
                .toggleClass("navigator-issue-only", options.issue)
                .toggleClass("page-type-navigator", options.search);
        },

        hide: function() {
            this._setBodyClasses({
                error: false,
                issue: false,
                search: true
            });

            if (this.isVisible()) {
                this.searchContainer.insertBefore(this.issueContainer);
                this.issueContainer.detach();
                this.trigger("issueHidden");
            }
        },

        updatePager: function(pager) {
            Application.execute("pager:update", pager);
        },

        /**
         * Load and show an issue.
         *
         * @param {object} issue The issue to show.
         * @param {number} issue.id The issue's ID.
         * @param {string} issue.key The issue's key.
         * @return {jQuery.Deferred} A deferred that is resolved when the issue is visible.
         */
        show: function(issue) {
            this.activate();
            Application.execute("issueEditor:setContainer", this.issueContainer);
            return Application.request("issueEditor:loadIssue", issue).always(_.bind(function () {
                jQuery(".js-stalker", this.issueContainer).stalker();

                this.updatePager(issue.pager);
                // now is a good time to pre-fetch anything that needs to be pre-fetched
                if (this.issueCacheManager) {
                    this.issueCacheManager.prefetchIssues();
                }
            }, this));
        },

        isLoading: function() {
            return Application.request("issueEditor:isCurrentlyLoading");
        },

        registerIssueCacheManager: function(issueCacheManager) {
            this.issueCacheManager = issueCacheManager;
        }
    });
});
