AJS.test.require([
    "com.atlassian.jira.jira-issue-nav-plugin:testutils",
    "com.atlassian.jira.jira-issue-nav-plugin:issuenav",
    "com.atlassian.jira.jira-issue-nav-plugin:metrics"
], function() {
    "use strict";

    module("JIRA.Issues.enhanceLinks", {
        setup: function() {
            //Stop inline layer from appending to window.top so qunit iframe runner works
            AJS.params.ignoreFrame = true;
            this.sandbox = sinon.sandbox.create();
            this.$testElement = AJS.$("<div></div>");
            AJS.$("body").append(this.$testElement);

            this.oldBackboneHistory = Backbone.history;
            Backbone.history = {
                options: {
                    root: ""
                }
            };

            this.router = {
                navigate: sinon.spy()
            };

            this.sandbox.spy(JIRA.Issues.Application, 'execute');
            this.searchPageModule = JIRA.Issues.TestUtils.mockSearchPageModule();
            this.searchPageModule.searchResults = JIRA.Issues.TestUtils.mockSearchResults();
            this.searchPageModule.update = sinon.spy();
            this.searchPageModule.reset = sinon.spy();

            this.documentEventHandlerRollback = beginEventHandlerTransaction(document);
        },
        teardown: function() {
            Backbone.history = this.oldBackboneHistory;
            this.sandbox.restore();
            this.$testElement.remove();
            this.router.navigate.reset();
            this.searchPageModule.update.reset();
            this.searchPageModule.reset.reset();

            this.documentEventHandlerRollback();  // test might have created event handlers on document; let's chuck them away shall we? :)
        }
    });

    function beginEventHandlerTransaction(target) {
        var bindings = [];
        var add = AJS.$.event.add;
        AJS.$.event.add = function(element, types, fn, data, selector) {
            if (element === target) {
                bindings.push([element, types, fn, selector]);
            }
            add.apply(AJS.$.event, arguments);
        };
        return function() {
            bindings.forEach(function(binding) {
                AJS.$.event.remove.apply(AJS.$.event, binding);
            });
        };
    }

    test("withPushState", function() {
        var $link = AJS.$('<a href="/issues/?query=123#fragment=test" class="push-state">link</a>');
        this.$testElement.append($link);

        JIRA.Issues.enhanceLinks.withPushState({});
        $link.click();

        ok(JIRA.Issues.Application.execute.calledOnce,
            "NavigationController is called"
        );

        ok(JIRA.Issues.Application.execute.calledWith('navigation:navigateToUrl', "/issues/?query=123"),
            "Clicked queryString is passed to NavigationController without fragment part of the URL"
        );
    });

    test("Links pass correct navigate params depending on placement & SearchPageModule presence", function() {
        JIRA.Issues.enhanceLinks.toIssue();

        var $normalCell = jQuery("<td><a href='#' class='issue-link my-link' data-issue-key='HSP-1'>test</a></td>");
        var $issueTableFrag = jQuery("<div class='list-view'><table id='issuetable'><tr></tr></table></div>");

        $issueTableFrag.find("tr:first").append($normalCell);

        $issueTableFrag.appendTo("#qunit-fixture");
        $normalCell.find(".my-link").click();

        ok(JIRA.Issues.Application.execute.calledWithExactly('navigation:navigate', {"selectedIssueKey": "HSP-1"}, {reset: true}));

        JIRA.Issues.enhanceLinks.registerSearchPageModule(this.searchPageModule);
        $normalCell.find(".my-link").click();

        ok(JIRA.Issues.Application.execute.calledThrice);
        ok(JIRA.Issues.Application.execute.calledWith('analytics:trigger', "kickass.openIssueFromTable"));
        ok(JIRA.Issues.Application.execute.calledWithExactly('navigation:navigate', {"selectedIssueKey": "HSP-1"}));
    });

    test("Links inside dropdown of issue table call update", function() {
        JIRA.Issues.enhanceLinks.toIssue();

        var $dropdown = jQuery("<td><a href='#' class='my-dropdown'>dropdown</a><ul class='aui-list'><li><a href='#' class='issue-link my-dropdown-link' data-issue-key='HSP-2'>dropdown test</a></li></ul></td>");
        var $issueTableFrag = jQuery("<div class='list-view'><table id='issuetable'><tr></tr></table></div>");

        $issueTableFrag.find("tr:first").append($dropdown);


        var $issueLink = $dropdown.find(".my-dropdown-link");

        new AJS.Dropdown({
            trigger: $dropdown.find(".my-dropdown"),
            content: $dropdown.find(".aui-list")
        });

        $issueTableFrag.appendTo("#qunit-fixture");
        $dropdown.find(".my-dropdown").click();
        $issueLink.click();

        ok(JIRA.Issues.Application.execute.calledTwice);
        ok(JIRA.Issues.Application.execute.calledWith('navigation:navigate', {"selectedIssueKey": "HSP-2"}));
    });

});
