(function() {
    "use strict";

    AJS.namespace("JIRA.Issues.IssueNavRouter");

    JIRA.Issues.IssueNavRouter = JIRA.Issues.BaseRouter.extend({

        initialize: function(options) {
            _.extend(this, options);
            _.bindAll(this,
                "_restoreSessionSearch",
                "_route");

            this.route(/^(.*?)([\?]{1}.*)?$/, this._route);
            this.route(/^(issues\/)?$/, this._restoreSessionSearch);

            // backbone-query-parameters supports clever decoding of values into arrays, but we don't want this.
            delete Backbone.Router.arrayValueSplit;

            // listen to changes of state
            this.listenTo(JIRA.Issues.Application, 'navigation:stateChanged', this.onStateChanged);
        },

        /**
         * Overwrite JIRA.Issues.BaseRouter, now it fires an event each time the URL changes
         */
        navigate: function() {
            JIRA.Issues.Tipsy.revalidate();
            this.trigger("navigate");
            JIRA.Issues.BaseRouter.prototype.navigate.apply(this, arguments);
        },

        onStateChanged: function onStateChanged(state, options) {
            //state changed - perform navigation to keep url up to date
            options = options || {};

            if (options.forceFullPageLoad) {
                // defer this until other 'stateChanged' event handlers had a chance to run
                // this assumes that those event handlers do their work synchronously
                _.defer(this._redirectToUrl, state.toUrl());
            } else {
                var navigateOptions = {
                    trigger: false,
                    replace: options.replace || false
                };
                this._setStatePermalink(state);
                this.navigate(state.toUrl(), navigateOptions);
            }
        },

        /**
         * Navigate to a new state.
         *
         * @param {JIRA.Issues.URLSerializer.state} state
         * @deprecated use JIRA.Issues.Application.execute("navigation:navigate") instead
         */
        pushState: function(state) {
            this._setStatePermalink(state);
            this.navigate(JIRA.Issues.URLSerializer.getURLFromState(state), {trigger: false});
        },

        /**
         * @deprecated use JIRA.Issues.Application.execute("navigation:navigate", {replace: true}) instead
         */
        replaceState: function(state) {
            this._setStatePermalink(state);
            this.navigate(JIRA.Issues.URLSerializer.getURLFromState(state), {trigger: false, replace: true});
        },

        _restoreSessionSearch: function() {
            var sessionSearch = this.initialSessionSearchState;
            JIRA.Issues.Application.execute('navigation:updateState', sessionSearch, {replace: true, reset: true, resetQuery: true});
        },

        /**
         * The "catch-all" route that distinguishes search and issue fragments.
         *
         * @param {string} path The path component of the URL (relative to the root)
         * @param {object} query The decoded querystring params
         * @private
         */
        _route: function(path, query) {
            // Re-encode back to a full fragment, since we do our own parsing in NavigationController
            var fragment = this.toFragment(path, query);

            if (JIRA.Issues.ignorePopState) {
                // Workaround for Chrome bug firing a null popstate event on page load.
                // Backbone should fix this!
                // @see http://code.google.com/p/chromium/issues/detail?id=63040
                // @see also JRADEV-14804
                return;
            }

            var jqlFromServer = AJS.Meta.get('jql');
            if (jqlFromServer != null) {
                AJS.Meta.set('jql', null);
            }
            var navigateOptions = {checkPermission: true, replace: true, reset: true, routerEvent: true};
            JIRA.Issues.Application.execute('navigation:navigateToUrl', fragment, navigateOptions,
                    jqlFromServer != null ? {jql: jqlFromServer} : undefined);
        },
        /**
         * Set the permalink for a given state into AJS.Meta to be rendered by the share plugin
         */
        _setStatePermalink: function(state) {
            var viewIssueState = _.pick(state, "selectedIssueKey");
            var baseUrl = AJS.Meta.get("jira-base-url");
            if (!_.isEmpty(viewIssueState)) {
                AJS.Meta.set("viewissue-permlink",
                    baseUrl + "/" + JIRA.Issues.URLSerializer.getURLFromState(viewIssueState)
                );
            }
            var issueNavState = _.omit(state, "selectedIssueKey");
            if (!_.isEmpty(issueNavState)) {
                AJS.Meta.set("issuenav-permlink",
                    baseUrl + "/" + JIRA.Issues.URLSerializer.getURLFromState(issueNavState)
                );
            }
        },
        _redirectToUrl: function _redirectToUrl(url) {
            window.location.assign(JIRA.Issues.IssueNavRouter.getBackboneRoot() + url);
        }
    }, {
        /**
         * @return The backbone history root that should be used for the current browser.
         */
        getBackboneRoot: function() {
            return AJS.contextPath() + "/";
        }
    });

}());
