AJS.test.require(["com.atlassian.jira.jira-issue-nav-plugin:filters-component-test"], function () {
    "use strict";

    var _ = require("underscore");
    var Meta = require("jira/util/data/meta");
    var Filter = require("jira/issues/components/filters/entities/filter");
    var CopyDialog = require("jira/issues/components/filters/views/dialogs/copy");
    var SaveTemplateTestCommon = require("jira/issues/components/filters/views/dialogs/save-template-test-common");

    module('jira/issues/components/filters/views/dialogs/copy', {
        setup: function () {
            SaveTemplateTestCommon.setup();
        },

        teardown: function () {
            SaveTemplateTestCommon.teardown();
        }
    });

    test("It should check if the default sharing for the user is GLOBAL", function () {
        var dialog = new CopyDialog({
            model: new Filter({
                name: "test filter"
            })
        });
        SaveTemplateTestCommon.renderDialog(dialog);

        var lastRequest = _.last(SaveTemplateTestCommon.server.requests);
        ok(lastRequest.url.match("/rest/api/2/filter/defaultShareScope"), "Request to the right REST endpoint");
        equal(lastRequest.method, "GET", "Request uses GET method");
    });

    test("It should display a switch-to-private checkbox If the default sharing for the user is GLOBAL", function () {
        var dialog = new CopyDialog({
            model: new Filter({
                name: "test filter"
            })
        });
        SaveTemplateTestCommon.renderDialog(dialog, "GLOBAL");

        ok(dialog.dialog.$form.find("#setSharingPrivate:checkbox"), "The form contains a checkbox for sharingPrivate");
        equal(dialog.dialog.$form.find("label[for=setSharingPrivate]").text(), "issuenav.filters.sharing.default.changetoprivate",
            "The form contains a text about changing the default share scope to private");
    });

    test("It should not display a switch-to-private checkbox If the default sharing for the user is PRIVATE", function () {
        var dialog = new CopyDialog({
            model: new Filter({
                name: "test filter"
            })
        });
        SaveTemplateTestCommon.renderDialog(dialog, "PRIVATE");

        equal(dialog.dialog.$form.find("#setSharingPrivate:checkbox").length, 0, "The form does not contain a checkbox for sharingPrivate");
    });

    test("It should pass the name when submitting the form", function () {
        var dialog = new CopyDialog({
            model: new Filter({
                name: "test filter"
            })
        });
        SaveTemplateTestCommon.renderDialog(dialog);
        SaveTemplateTestCommon.submitDialog(dialog, {
            name: "This filter is a copy"
        });

        equal(JSON.parse(_.last(SaveTemplateTestCommon.server.requests).requestBody).name, "This filter is a copy");
    });

    test("It should pass the original filter id when submitting the form", function () {
        var dialog = new CopyDialog({
            model: new Filter({
                name: "test filter",
                id: "1234"
            })
        });
        SaveTemplateTestCommon.renderDialog(dialog);
        SaveTemplateTestCommon.submitDialog(dialog);

        equal(JSON.parse(_.last(SaveTemplateTestCommon.server.requests).requestBody).id, "1234");
    });

    test("It should copy the filter as favourite when submitting the form", function () {
        var dialog = new CopyDialog({
            model: new Filter({
                name: "test filter",
                id: "1234"
            })
        });
        SaveTemplateTestCommon.renderDialog(dialog);
        SaveTemplateTestCommon.submitDialog(dialog);

        equal(JSON.parse(_.last(SaveTemplateTestCommon.server.requests).requestBody).favourite, true);
    });

    test("It should display the 'Copy' header when rendering the form", function () {
        var dialog = new CopyDialog({
            model: new Filter({
                name: "test filter",
                id: "1234"
            })
        });
        SaveTemplateTestCommon.renderDialog(dialog);

        equal(dialog.dialog.$popup.find("h2").text(), "issue.nav.filters.copyialog.title : test filter");
    });

    test("It should prefill the filters name with 'Copy of ...'", function () {
        this.stub(AJS, "format");
        var dialog = new CopyDialog({
            model: new Filter({
                name: "test filter",
                id: "1234"
            })
        });
        SaveTemplateTestCommon.renderDialog(dialog);

        sinon.assert.calledWith(AJS.format, "common.words.copyof", "test filter");
    });

    test("It should display a message when copying a filter from another user", function () {
        this.stub(Meta, "get").withArgs("remote-user").returns("myUser");
        this.stub(AJS, "format");

        var dialog = new CopyDialog({
            model: new Filter({
                name: "test filter",
                id: "1234",
                ownerUserName: 'ownerUser',
                ownerDisplayName: "Mr. filter's owner"
            })
        });
        SaveTemplateTestCommon.renderDialog(dialog);

        sinon.assert.calledWith(AJS.format, "issue.nav.filters.savedialog.desc.otherowner", "Mr. filter's owner");
    });

    test("It should change user's default sharing to private when the switch-to-private checkbox is marked", function () {
        var dialog = new CopyDialog({
            model: new Filter({
                name: "test filter",
                id: "1234"
            })
        });
        SaveTemplateTestCommon.renderDialog(dialog, "GLOBAL");
        SaveTemplateTestCommon.submitDialog(dialog, {
            changeToPrivate: true
        });

        var lastRequest = _.last(SaveTemplateTestCommon.server.requests);
        deepEqual(JSON.parse(lastRequest.requestBody), {scope: "PRIVATE"}, "Request payload is valid");
        ok(lastRequest.url.match("/rest/api/2/filter/defaultShareScope"), "Request to the right REST endpoint");
        equal(lastRequest.method, "PUT", "Request uses PUT method");
    });

    test("It should focus the name field when there is an error submitting the form", function () {
        var dialog = new CopyDialog({
            model: new Filter({
                name: "test filter",
                id: "1234"
            })
        });
        SaveTemplateTestCommon.renderDialog(dialog);

        SaveTemplateTestCommon.submitDialog(dialog, {
            responseCode: 500
        });

        ok(document.activeElement === dialog.dialog.$popup.find("#filterName")[0]);
    });


    test("Check all cases for changing the submit button values", function () {
        SaveTemplateTestCommon.invokeSubmitButtonTests(function () {
            return SaveTemplateTestCommon.getCopySubmitButton();
        });
    });

    test("Check all cases for showing/not showing empty JQL warning", function () {
        SaveTemplateTestCommon.invokeEmptyJqlWarningTests(function (jqlString) {
            return SaveTemplateTestCommon.getCopyEmptyJqlWarning(jqlString);
        });
    });
});
