AJS.test.require(["com.atlassian.jira.jira-issue-nav-plugin:filters-component-test"], function () {
    "use strict";

    var _ = require("underscore");
    var Filter = require("jira/issues/components/filters/entities/filter");
    var Filters = require("jira/issues/components/filters/entities/filters");
    var TestUtils = require("jira/issues/test/utils");

    var ListWithActions;
    module('jira/issues/components/filters/controllers/list-with-actions', {
        setup: function() {
            this.sandbox = sinon.sandbox.create();
            var context = AJS.test.mockableModuleContext();

            this.mocks = {
                Actions: TestUtils.mockModule(this.sandbox, context, 'jira/issues/components/filters/views/actions'),
                ListWithActions: TestUtils.mockModule(this.sandbox, context, 'jira/issues/components/filters/views/list/list-with-actions')
            };
            // These two methods should not run, they require a ton of mocks. So we use a stub instead of a spy
            this.mocks.Actions.prototype.render.restore();
            this.mocks.Actions.prototype.close.restore();
            this.sandbox.stub(this.mocks.Actions.prototype, "render");
            this.sandbox.stub(this.mocks.Actions.prototype, "close");

            ListWithActions = context.require('jira/issues/components/filters/controllers/list-with-actions');
        },

        teardown: function() {
            this.sandbox.restore();
        },

        createAndOpenController: function () {
            var collection = new Filters(_.toArray(arguments), {fetchState: "fetched"});
            var controller = new ListWithActions({
                collection: collection
            });

            controller.show();

            return controller;
        },

        createModel: function (id, name) {
            return new Filter({id: id || "1234", name: name || "My filter"});
        },

        assertEvent: function (eventName) {
            var model = this.createModel();
            var controller = this.createAndOpenController(model);
            var listView = this.mocks.ListWithActions.thisValues[0];
            var eventSpy = this.spy();

            controller.on(eventName, eventSpy);
            listView.trigger("itemview:openActions", null, {view: null, model: model});

            var actionsView = this.mocks.Actions.thisValues[0];
            actionsView.trigger(eventName, {model: model});

            sinon.assert.calledOnce(actionsView.close);
            sinon.assert.calledOnce(eventSpy);
            sinon.assert.calledWith(eventSpy, model);
        }
    });

    test("When clicking on a actions trigger, it opens the actions view", function () {
        var model = this.createModel();
        this.createAndOpenController(model);
        var listView = this.mocks.ListWithActions.thisValues[0];

        listView.trigger("itemview:openActions", null, {view: null, model: model});

        ok(this.mocks.Actions.calledWithNew(), "A new actions view is created");
        equal(this.mocks.Actions.firstCall.args[0].model, model, "The model is passed to the actions view");
        sinon.assert.calledOnce(this.mocks.Actions.prototype.render, "Actions view is opened");
    });

    test("When clicking on the trigger twice, it closes the actions view", function () {
        var model = this.createModel();
        this.createAndOpenController(model);
        var listView = this.mocks.ListWithActions.thisValues[0];

        listView.trigger("itemview:openActions", null, {view: null, model: model});
        sinon.assert.calledOnce(this.mocks.Actions, "One view is created");
        sinon.assert.calledOnce(this.mocks.Actions.prototype.render, "The view is rendered");
        this.mocks.Actions.reset();

        listView.trigger("itemview:openActions", null, {view: null, model: model});
        sinon.assert.notCalled(this.mocks.Actions, "The view is not created again");
        sinon.assert.calledOnce(this.mocks.Actions.prototype.close, "The view is closed");
    });

    test("When clicking on the trigger for another filter, it opens the new actions view", function () {
        var model1 = this.createModel("1", "My first filter");
        var model2 = this.createModel("2", "My second filter");
        this.createAndOpenController(model1, model2);
        var listView = this.mocks.ListWithActions.thisValues[0];

        listView.trigger("itemview:openActions", null, {view: null, model: model1});
        listView.trigger("itemview:openActions", null, {view: null, model: model2});

        sinon.assert.calledTwice(this.mocks.Actions, "Two views are created");
        sinon.assert.calledOnce(this.mocks.Actions.prototype.close, "The view is closed");
        sinon.assert.calledTwice(this.mocks.Actions.prototype.render, "The view is rendered");
    });

    test("When closing the view, it closes the actions view too", function () {
        var model = this.createModel();
        var controller = this.createAndOpenController(model);
        var listView = this.mocks.ListWithActions.thisValues[0];

        listView.trigger("itemview:openActions", null, {view: null, model: model});
        controller.close();

        sinon.assert.calledOnce(this.mocks.Actions.prototype.close);
    });

    test("When selecting a filter, it closes the actions view", function () {
        var model = this.createModel();
        this.createAndOpenController(model);
        var listView = this.mocks.ListWithActions.thisValues[0];

        listView.trigger("itemview:openActions", null, {view: null, model: model});
        listView.trigger("itemview:selectFilter", null, {view: null, model: model});

        sinon.assert.calledOnce(this.mocks.Actions.prototype.close);
    });

    test("When highlighting a filter, it closes the actions view", function () {
        var model = this.createModel();
        var controller = this.createAndOpenController(model);
        var listView = this.mocks.ListWithActions.thisValues[0];

        listView.trigger("itemview:openActions", null, {view: null, model: model});
        controller.highlightFilter(model);

        sinon.assert.calledOnce(this.mocks.Actions.prototype.close);
    });

    test("It rethrows the deleteFilter event from the actions view, and closes it", function () {
        this.assertEvent("deleteFilter");
    });

    test("It rethrows the renameFilter event from the actions view, and closes it", function () {
        this.assertEvent("renameFilter");
    });

    test("It rethrows the copyFilter event from the actions view, and closes it", function () {
        this.assertEvent("copyFilter");
    });

    test("It rethrows the unfavouriteFilter event from the actions view, and closes it", function () {
        this.assertEvent("unfavouriteFilter");
    });
});
