AJS.test.require(["com.atlassian.jira.jira-issue-nav-plugin:filters-component-test"], function () {
    "use strict";

    var Filters = require("jira/issues/components/filters/entities/filters");
    var TestUtils = require("jira/issues/test/utils");
    var ListController;

    module('jira/issues/components/filters/controllers/list', {
        CLASS_NAME: "my-saved-filters",

        setup: function() {
            this.sandbox = sinon.sandbox.create();
            var context = AJS.test.mockableModuleContext();

            this.mocks = {
                ModuleView: TestUtils.mockModule(this.sandbox, context, 'jira/issues/components/filters/views/list/module'),
                ListView: TestUtils.mockModule(this.sandbox, context, 'jira/issues/components/filters/views/list/list'),
                MessageView: TestUtils.mockModule(this.sandbox, context, 'jira/issues/components/filters/views/list/message'),
                LoginUtils: TestUtils.mockModule(this.sandbox, context, 'jira/issues/utils/login')
            };

            ListController = context.require('jira/issues/components/filters/controllers/list');
        },

        teardown: function() {
            this.sandbox.restore();
        },

        newFilterWithId: function(idAsString) {
            return {
                id: idAsString,
                name: idAsString
            };
        },

        newEmptyFetchedCollection: function() {
            return new Filters([], {fetchState: "fetched"});
        },

        newFetchedCollectionWithAFilter: function(filter) {
            return this.newEmptyFetchedCollection().add(filter);
        }
    });

    test("It renders the module view", function() {
        var controller = new ListController({
            collection: new Filters([])
        });

        controller.show();

        sinon.assert.calledOnce(this.mocks.ModuleView.prototype.render);
    });

    test("It renders the module view with a title", function() {
        var expectedModuleTitle = "Module title";
        var controller = new ListController({
            collection: new Filters([]),
            title: expectedModuleTitle
        });

        controller.show();

        equal(this.mocks.ModuleView.firstCall.args[0].title, expectedModuleTitle);
    });

    test("When the collection as been fetched with an error, it renders a message with the error text", function() {
        var expectedErrorMessage = "There is an error";
        var controller = new ListController({
            collection: new Filters([], {fetchState: "error"}),
            errorMessage: expectedErrorMessage,
            className: this.CLASS_NAME
        });

        controller.show();

        sinon.assert.calledOnce(this.mocks.MessageView.prototype.render);
        equal(this.mocks.MessageView.firstCall.args[0].text, expectedErrorMessage);
        equal(this.mocks.MessageView.firstCall.args[0].className, this.CLASS_NAME);
    });

    test("When the collection has no fetch state, it renders a message with the loading text", function() {
        var expectedLoadingMessage = "Loading...";
        var controller = new ListController({
            collection: new Filters([]),
            loadingMessage: expectedLoadingMessage,
            className: this.CLASS_NAME
        });

        controller.show();

        sinon.assert.calledOnce(this.mocks.MessageView.prototype.render);
        equal(this.mocks.MessageView.firstCall.args[0].text, expectedLoadingMessage);
        equal(this.mocks.MessageView.firstCall.args[0].className, this.CLASS_NAME);
    });

    test("When the collection has been fetched and it is empty, it renders a message with the empty text", function() {
        this.stub(this.mocks.LoginUtils, "isLoggedIn").returns(true);
        var expectedEmptyMessage = "Nothing here";
        var controller = new ListController({
            collection: this.newEmptyFetchedCollection(),
            emptyMessage: expectedEmptyMessage,
            className: this.CLASS_NAME
        });

        controller.show();

        sinon.assert.calledOnce(this.mocks.MessageView.prototype.render);
        equal(this.mocks.MessageView.firstCall.args[0].text, expectedEmptyMessage);
        equal(this.mocks.MessageView.firstCall.args[0].className, this.CLASS_NAME);
    });

    test("When rendering the empty message for an anonymous user, it renders a message with the login text", function() {
        this.stub(this.mocks.LoginUtils, "isLoggedIn").returns(false);
        var expectedLoginMessage = "Go to login";
        var controller = new ListController({
            collection: this.newEmptyFetchedCollection(),
            loginMessage: expectedLoginMessage,
            className: this.CLASS_NAME
        });

        controller.show();

        sinon.assert.calledOnce(this.mocks.MessageView.prototype.render);
        equal(this.mocks.MessageView.firstCall.args[0].text, expectedLoginMessage);
        equal(this.mocks.MessageView.firstCall.args[0].className, this.CLASS_NAME);
    });

    test("When rendering the empty message, it renders the list if an item is added to the collection", function() {
        this.stub(this.mocks.LoginUtils, "isLoggedIn").returns(false);
        var collection = this.newEmptyFetchedCollection();
        var controller = new ListController({
            collection: collection,
            className: this.CLASS_NAME
        });

        controller.show();
        collection.add(this.newFilterWithId("1234"));

        sinon.assert.calledOnce(this.mocks.ListView.prototype.render);
    });

    test("When the collection as been fetched with content, it renders the list view", function() {
        var collection = this.newFetchedCollectionWithAFilter(this.newFilterWithId("1234"));
        var controller = new ListController({
            collection: collection,
            className: this.CLASS_NAME
        });

        controller.show();

        sinon.assert.calledOnce(this.mocks.ListView.prototype.render);
        equal(this.mocks.ListView.firstCall.args[0].collection, collection);
        equal(this.mocks.ListView.firstCall.args[0].className, this.CLASS_NAME);
    });

    test("When the collection has been fetched with content and the list view renders, a 'list:render' event is triggered", function() {
        var controller = new ListController({
            collection: this.newFetchedCollectionWithAFilter(this.newFilterWithId("1234")),
            className: this.CLASS_NAME
        });
        var listRenderSpy = this.spy();
        controller.on("list:render", listRenderSpy);

        controller.show();

        sinon.assert.calledOnce(listRenderSpy);
    });

    test("When the collection as been fetched with content, it renders the empty view when all items from the collection are removed", function() {
        this.stub(this.mocks.LoginUtils, "isLoggedIn").returns(true);

        var expectedEmptyMessage = "Nothing here";
        var anyFilter = this.newFilterWithId("1234");
        var collection = this.newFetchedCollectionWithAFilter(anyFilter);
        var controller = new ListController({
            collection: collection,
            emptyMessage: expectedEmptyMessage,
            className: this.CLASS_NAME
        });

        controller.show();
        collection.remove(anyFilter);

        sinon.assert.calledOnce(this.mocks.MessageView.prototype.render);
        equal(this.mocks.MessageView.firstCall.args[0].text, expectedEmptyMessage);
        equal(this.mocks.MessageView.firstCall.args[0].className, this.CLASS_NAME);
    });

    test("When a list is initially rendered, it triggers the 'render' event", function() {
        var controller = new ListController({
            collection: this.newFetchedCollectionWithAFilter(this.newFilterWithId("1234"))
        });
        var onRender = this.spy();

        controller.on("render", onRender);
        controller.show();

        sinon.assert.calledOnce(onRender);
    });

    test("When a new item is rendered, it triggers the 'render' event", function() {
        var collection = this.newFetchedCollectionWithAFilter(this.newFilterWithId("1234"));
        var controller = new ListController({
            collection: collection
        });
        var onRender = this.spy();

        controller.show();
        controller.on("render", onRender);
        collection.add(this.newFilterWithId("4567"));

        sinon.assert.calledOnce(onRender);
    });

    test("When a filter is selected, it triggers the 'selectFilter' event", function() {
        var expectedFilter = this.newFilterWithId("1234");
        var controller = new ListController({
            collection: this.newFetchedCollectionWithAFilter(expectedFilter)
        });
        var onSelectFilter = this.spy();
        controller.on("selectFilter", onSelectFilter);
        controller.show();
        var viewInstance = this.mocks.ListView.thisValues[0];

        viewInstance.trigger("itemview:selectFilter", viewInstance, {model: expectedFilter});

        sinon.assert.calledOnce(onSelectFilter);
        sinon.assert.calledWith(onSelectFilter, expectedFilter);
    });
});
