(function() {
	AJS.namespace("JIRA.Importer.Trello.ProjectSelectionFormView");
	JIRA.Importer.Trello.ProjectSelectionFormView = Backbone.View.extend({
		events: {
			"keyup input.project-name-field": "onProjectNameUpdate",
			"change input.project-name-field": "onProjectNameChange",
			"keyup input.project-key-field": "onProjectKeyUpdate",
			"change input.project-key-field": "onProjectKeyChange",
			"click input[type=checkbox]": "onProjectEnable"
		},

		initialize: function () {
			// Pre-generate the project keys (but only if they aren't already there).
			_.each(this.$el.find(".project-name-field"), function (nameInput) {
				var $nameInput = AJS.$(nameInput);

				var $keyInput = this._getProjectKeyField($nameInput);
				var generatedKey = this.generateProjectKey($nameInput);
				if (AJS.$.trim($keyInput.val()).length == 0) {
					$keyInput.val(generatedKey);
				} else if ($keyInput.val() !== generatedKey) {
					$keyInput.data("manuallyChanged", true);
				}
			}, this);

			// Save error messages associated with individual inputs.
			this.initialErrors = {};
			this.errors = {};
			_.each(this.$el.find("input"), function(inputElement) {
				var $inputElement = AJS.$(inputElement);
				var $errorDiv = this._getErrorDiv($inputElement);
				if ($errorDiv.length > 0 && AJS.$.trim($errorDiv.text()).length > 0) {
					this.addError($inputElement, AJS.$.trim($errorDiv.text()));
					this.initialErrors[$inputElement.attr("name")] = AJS.$.trim($errorDiv.text());
				}
			}, this);

			// Pre-disable unchecked elements.
			_.each(this.$el.find("input[type=checkbox]"), function(e) {
				this._toggleBoard(AJS.$(e));
			}, this);

			this.render();
		},

		render: function() {
			// Restrict duplicate names/keys which may have been generated.
			var $nameInputs = this._getAllInputs("project-name-field");
			this.restrictDuplicateValues($nameInputs, AJS.I18n.getText("com.atlassian.jira.plugins.importer.trello.trelloProjectSelectionPage.error.duplicateProjectName"));

			var $keyInputs = this._getAllInputs("project-key-field");
			this.restrictDuplicateValues($keyInputs, AJS.I18n.getText("com.atlassian.jira.plugins.importer.trello.trelloProjectSelectionPage.error.duplicateProjectKey"));

			_.each(_.keys(this.errors), function(fieldName) {
				// Display the errors in the error div.
				var errorContent = _.keys(this.errors[fieldName]).join(" ");
				var $field = this.$el.find("input[name=" + fieldName + "]");
				this._getErrorDiv($field).text(errorContent);

				// Set the field to an error field.
				if (!$field.attr("disabled") && errorContent.length > 0) {
					$field.addClass("error");
				}
			}, this);

			// If there were errors, disable the input button. Also disable it if there are no enabled inputs.
			if (this.$el.find("input.error:not(:disabled)").filter(":visible").length > 0 || this._getAllInputs("project-name-field").length == 0) {
				AJS.$("#nextButton").attr("disabled", "disabled");
			} else {
				AJS.$("#nextButton").attr("disabled", null);
			}
		},

		/**
		 * Generate a project key for the given project nae input.
		 */
		generateProjectKey: function($nameInput) {
			return new JIRA.ProjectKeyGenerator().generateKey($nameInput.val());
		},

		/**
		 * Restrict all fields within $otherFields to have unique names. If they don't, display the error `error` on each field.
		 */
		restrictDuplicateValues: function($fields, error) {
			_.each($fields, function (field) {
				var $field = AJS.$(field);

				// Find all elements which conflict with this field's value.
				var $conflictingFields = _.filter($fields, function(field) {
					return AJS.$(field).val() == $field.val();
				});

				// If there are fields in conflict, save the error to each conflicting field.
				if ($conflictingFields.length > 1) {
					_.each($conflictingFields, function(field) {
						this.addError(AJS.$(field), error);
					}, this);
				} else {
					// Otherwise, make sure this error doesn't exist within this field.
					this.removeError($field, error);
				}
			}, this);
		},

		/**
		 * Event functions.
		 */
		onProjectNameUpdate: function(event) {
			// If the key hasn't been changed manually, generate the key.
			var $nameInput = AJS.$(event.currentTarget);
			var $keyInput = this._getProjectKeyField($nameInput);
			if (!$keyInput.data("manuallyChanged"))
				$keyInput.val(this.generateProjectKey($nameInput)).change();

			// Remove any server-side errors.
			var $keyInput = AJS.$(event.currentTarget);
			var inputName = $keyInput.attr("name");
			if (this.initialErrors[inputName]) {
				this.removeError($keyInput, this.initialErrors[inputName]);
				delete this.initialErrors[inputName];
			}

			this.render();
		},

		onProjectNameChange: function(event) {
			// Remove any server-side errors.
			var $nameInput = AJS.$(event.currentTarget);
			var inputName = $nameInput.attr("name");
			if (this.initialErrors[inputName]) {
				this.removeError($nameInput, this.initialErrors[inputName]);
				delete this.initialErrors[inputName];
			}

			this.render();
		},

		onProjectKeyUpdate: function(event) {
			// Save the fact that the key was changed.
			var $keyInput = AJS.$(event.currentTarget);
			$keyInput.data("manuallyChanged", true);

			// Keys must be in uppercase.
			$keyInput.val($keyInput.val().toUpperCase());

			// Remove any server-side errors.
			var inputName = $keyInput.attr("name");
			if (this.initialErrors[inputName]) {
				this.removeError($keyInput, this.initialErrors[inputName]);
				delete this.initialErrors[inputName];
			}

			this.render();
		},

		onProjectKeyChange: function(event) {
			// Remove any server-side errors.
			var $keyInput = AJS.$(event.currentTarget);
			var inputName = $keyInput.attr("name");
			if (this.initialErrors[inputName]) {
				this.removeError($keyInput, this.initialErrors[inputName]);
				delete this.initialErrors[inputName];
			}

			this.render();
		},

		onProjectEnable: function(event) {
			var $checkbox = this.$(event.currentTarget);
			this._toggleBoard($checkbox);
			this.render();
		},

		/**
		 * Utility methods.
		 */
		addError: function($field, error) {
			if (!error) {
				return;
			}

			var fieldName = $field.attr("name");
			if (this.errors[fieldName] === undefined) {
				this.errors[fieldName] = {};
			}

			if (!(error in this.errors[fieldName])) {
				this.errors[fieldName][error] = true;
			}
		},

		removeError: function($field, error) {
			var fieldName = $field.attr("name");
			if (this.errors[fieldName] === undefined) {
				return;
			}

			if (error in this.errors[fieldName]) {
				delete this.errors[fieldName][error];

				if (_.keys(this.errors[fieldName]).length === 0) {
					$field.removeClass("error");
				}
			}
		},

		hasErrors: function() {
			return AJS.$("#nextButton").attr("disabled") === "disabled";
		},

		_getProjectKeyField: function($projectNameField) {
			return $projectNameField.parentsUntil("table", "tr").find("input.project-key-field");
		},

		_getProjectNameField: function($projectSelectionCheckbox) {
			return $projectSelectionCheckbox.parentsUntil("table", "tr").find("input.project-name-field");
		},

		_getErrorDiv: function($field) {
			return $field.next("div.error");
		},

		_getAllInputs: function(cls) {
			return this.$el.find("input." + cls + ":not(:disabled)");
		},

		_toggleBoard: function($checkbox) {
			var $projectNameField = this._getProjectNameField($checkbox);
			var $projectKeyField = this._getProjectKeyField($projectNameField);

			// If the checkbox was disabled, disable the corresponding inputs too.
			if (!$checkbox.attr("checked")) {
				$projectNameField.attr("disabled", "disabled");
				this._getErrorDiv($projectNameField).hide();

				$projectKeyField.attr("disabled", "disabled");
				this._getErrorDiv($projectKeyField).hide();
			} else {
				$projectNameField.attr("disabled", null);
				$projectKeyField.attr("disabled", null);
				this._getErrorDiv($projectNameField).show();
				this._getErrorDiv($projectKeyField).show();
			}
		}
	});
}());

AJS.$(function() {
	var $jimForm = AJS.$("form#jimform");
	var form_view = new JIRA.Importer.Trello.ProjectSelectionFormView({
		el: AJS.$("#projectSelection")
	});

	// Remove the onsubmit in-HTML event which is placed on the JIM form, to ensure the following logic works.
	$jimForm.attr("onsubmit", null);

	// Make sure you cannot submit the form if the submit button is disabled.
	$jimForm.submit(function() {
		return !form_view.hasErrors();
	});
});
