require([
        'hipchat/admin/configure-integration-data'
    ],

    function (integrationData) {
        'use strict';

        AJS.toInit(function ($) {
            "use strict";

            /**
             * Hipchat configuration state
             * @type {String} The current configuration state of hipchat
             */
            var state = $("#hipchat-integration-configuration").val();

            var localStorage = window.localStorage;

            // If the state element is not present, do not load the rest as this is
            //   the wrong page
            if (state) {

                /**
                 * Routes used in directing requests/redirects
                 */
                var routes = {
                    self: $("#hipchat-route-config").val(),
                    serverPing: $("#hipchat-route-check-server-connectivity").val(),
                    clientPing: $("#hipchat-route-check-client-connectivity").val(),
                    beginInstall: $("#hipchat-route-begin-install").val(),
                    completeInstall: $("#hipchat-route-complete-install").val(),
                    installableUrl:  $("#hipchat-route-installable-url").val(),
                    uninstall: $("#hipchat-route-uninstall").val(),
                    redirectUrl: $("#hipchat-route-redirectUrl").val()
                };

                /**
                 * Resource urls
                 */
                var resources = {
                    screenshotResource: $("#hipchat-server-resource-screenshot").val(),
                    screenshotResource2x: $("#hipchat-server-resource-screenshot-2x").val(),
                    uninstalling: $("#hipchat-server-resource-uninstalling").val()
                };

                /**
                 * Misc other values/imports
                 */
                var Templates = HipChat.Templates.Configuration;
                var panel = $("#hipchat-configuration-panel");
                if (integrationData.linkId) {
                    var linkId = integrationData.linkId;
                }

                /**
                 * All views for this page
                 */
                var Views = (function () {
                    var confirmDialog;

                    return {
                        /**
                         * Load configure template
                         */
                        installing: function () {
                            $('#hipchat-installing .hipchat-install-message').spin('large');
                        },
                        /**
                         * Load error template
                         * @param msg {String} Error message to display
                         */
                        error: function (msg) {
                            panel.html(Templates.error({msg: msg}));
                        },
                        /**
                         * Load uninstall error template
                         */
                        uninstallError: function (msg) {
                            if (msg.status == 401) {
                                panel.html(Templates.uninstallPermissionError());
                            } else {
                                panel.html(Templates.uninstallConnectionError());
                            }
                        },
                        /**
                         * Confirms removal of the link to HipChat
                         *
                         * @return promise resolved if the user confirms removed, rejected if they cancel
                         */
                        confirmUninstall: function () {
                            var deferred = $.Deferred();

                            if (!confirmDialog) {
                                // TODO CONFDEV-28519 ADG 2 - new confirmation dialog (and remove related CSS)
                                confirmDialog = new AJS.Dialog(600, 300, "hipchat-remove-link-dialog");
                                confirmDialog.addHeader(AJS.I18n.getText("plugins.hipchat.admin.removelink.header"), "remove-warning");
                                confirmDialog.addPanel("Message Panel", Templates.removeLinkMessage());
                                confirmDialog.addButton(AJS.I18n.getText("plugins.hipchat.admin.removelink.confirm"), function () {
                                    confirmDialog.hide();
                                    Views.uninstall();
                                    deferred.resolve();
                                });
                                confirmDialog.addCancel(AJS.I18n.getText("plugins.hipchat.admin.removelink.cancel"), function () {
                                    confirmDialog.hide();
                                    deferred.reject();
                                });
                            }
                            confirmDialog.show();

                            return deferred.promise();
                        },
                        uninstall: function () {
                            panel.html(Templates.uninstalling({
                                uninstallingImage: resources.uninstalling
                            }));
                            $('#hipchat-uninstalling .hipchat-install-message').spin('large');
                        },
                        reinstallationSucceeded: function () {
                            try {
                                var flag = require('aui/flag');
                                flag({
                                    type: "success",
                                    title: AJS.I18n.getText("plugins.hipchat.reinstall.success.title"),
                                    persistent: false,
                                    body: AJS.I18n.getText("plugins.hipchat.reinstall.success.description")
                                });
                            } catch (e) {
                                // AUI flag is not available, do not bother.
                            }
                        }
                    };
                })();

                /**
                 * Remote APIs service
                 */
                var Remote = {
                    /**
                     * Begin a server request to ping the hipchat server
                     *  to ensure server is visible
                     * @returns {*}
                     */
                    serverPing: function () {
                        return $.ajax(routes.serverPing, {
                            type: "POST",
                            dataType: 'json',
                            contentType: 'application/json',
                            data: JSON.stringify({
                                serverUrl: ""
                            })
                        });
                    },

                    /**
                     * Server ping with a specific url
                     * @param url
                     * @returns {*}
                     */
                    serverPingToUrl: function (url) {
                        return $.ajax(routes.serverPing, {
                            type: "POST",
                            dataType: 'json',
                            contentType: 'application/json',
                            data: JSON.stringify({
                                serverUrl: url
                            })
                        });
                    },

                    /**
                     * Request information to begin installing a hipchat
                     *  plugin. This request also sets a cookie which is
                     *  used for XSRF protection
                     * @returns {*}
                     */
                    beginInstall: function (url, reinstall) {
                        return $.ajax(routes.beginInstall, {
                            type: "POST",
                            dataType: 'json',
                            contentType: 'application/json',
                            data: JSON.stringify({
                                installableUrl: url,
                                reinstall: reinstall
                            })
                        })
                    },
                    /**
                     * Submit the installation completion data to the server
                     * @param url
                     * @returns {*}
                     */
                    completeInstall: function (url) {
                        return $.ajax(routes.completeInstall, {
                            type: "POST",
                            dataType: 'json',
                            contentType: 'application/json',
                            data: JSON.stringify({
                                installableUrl: url
                            })
                        })
                    },
                    /**
                     * Remove link with the given id
                     * @param linkId
                     * @returns {*}
                     */
                    remove: function (linkId) {
                        return $.ajax({
                            url: routes.uninstall + "/" + linkId,
                            type: "POST",
                            dataType: 'json',
                            contentType: 'application/json',
                            data: JSON.stringify({})
                        })
                    },

                    reinstall: function (linkId) {
                        return $.ajax({
                            url: routes.uninstall + "/" + linkId,
                            type: "POST",
                            dataType: 'json',
                            contentType: 'application/json',
                            data: JSON.stringify({
                                reinstall: true
                            })
                        })
                    },

                    removeDefault: function () {
                        return $.ajax(routes.uninstall, {
                            type: "POST",
                            dataType: 'json',
                            contentType: 'application/json',
                            data: JSON.stringify({})
                        })
                    }
                };

                /**
                 * Code related to the various actions on this page
                 */
                var Actions = {
                    /**
                     * Setup the initial state of the screen
                     * @param state
                     */
                    initialize: function (state) {
                        switch (state) {
                            case "installing":
                                Actions.installing();
                                break;

                            case "installed":
                                Actions.installed(linkId);
                                break;

                            case "uninstalling":
                                Actions.uninstalling();
                                break;

                            case "reinstall":
                                Actions.reinstalling();
                                break;

                            case "configure":
                            default:
                                Actions.configure();
                        }
                    },

                    /**
                     * Utility function to redirect the client's browser (current url is not in history)
                     * @param url
                     */
                    redirect: function (url) {
                        window.location.replace(url);
                    },

                    /**
                     * Utility function to redirect the client's browser while keeping the current url in the browser history.
                     */
                    navigate: function (url) {
                        window.location.assign(url);
                    },

                    /**
                     * Load the configuration screen
                     *
                     * Trigger a server ping, if successful request
                     *  installation information and redirect to hipchat
                     *  plugin installation.
                     */
                    configure: function () {
                        Remote.serverPing()
                            .done(function () {
                                $("#hipchat-install").removeAttr("disabled");
                            })
                            .fail(function () {
                                $("#hipchat-server-failure").removeClass("hidden");
                            })
                            .always(function () {
                                $("#hipchat-loading").hide();
                            });

                        $("#hipchat-install").click(function (e) {
                            e.preventDefault();
                            if ($(this).attr('aria-disabled')) {
                                return;
                            }
                            $(this).attr('aria-disabled', 'true');

                            AJS.trigger('analyticsEvent', {
                                name: 'notifications.hipchat.server.registration.started',
                                data: {}
                            });

                            Remote.beginInstall(undefined, false)
                                .done(Actions.navigate);
                        });

                        $('#enable-btf-install').click(function (e) {
                            e.preventDefault();
                            window.onbeforeunload = null; // Prevent "Are you sure you want to navigate away" dialog to appear

                            enableHipChatBTF();
                        });

                        //Fire analytics event
                        AJS.trigger('analyticsEvent', {
                            name: 'notifications.hipchat.server.discovered',
                            data: {}
                        });

                        /**
                         * We enable the page to set a url
                         */
                        function enableHipChatBTF() {
                            $('.hipchat-install-form').toggleClass("hipchat-btf");
                            $('#btf-url').focus();
                            $('#btf-install').click(function (j) {
                                $("#btf-error").hide();
                                j.preventDefault();
                                var url = $('#btf-url').val();
                                var regex = /^https?:\/\//i;

                                // If the user did not put anything we put https on the url
                                if (!regex.test(url)) {
                                    url = 'https://' + url;
                                }

                                Remote.serverPingToUrl(url)
                                    .done(function () {
                                        Remote.beginInstall(url, false).done(Actions.navigate)
                                    })
                                    .fail(function () {
                                        $("#btf-error").show();
                                    });
                            });
                        }

                        $('#btf-install-cancel').click(function (e) {
                            e.preventDefault();
                            $('.hipchat-install-form').toggleClass("hipchat-btf");
                        });
                    },

                    /**
                     * Load the installing screen
                     *
                     * Posts the installation information to the server.
                     *
                     * Done using javascript as this lets the process be sort
                     *  of async, and the user gets some feedback on slow connections
                     *  instead of just seeing a white screen.
                     */
                    installing: function () {
                        Views.installing();
                        Remote.completeInstall(routes.installableUrl)
                            .done(function (data) {
                                Actions.redirect(routes.self);
                            }).error(Views.error)
                    },

                    /**
                     * Load the installed screen
                     */
                    installed: function () {
                        $("#hipchat-uninstall").click(function (e) {
                            e.preventDefault();
                            if ($(this).attr('aria-disabled')) {
                                return;
                            }
                            Views.confirmUninstall().done(function () {
                                Remote.remove(linkId).done(Actions.redirect).error(Views.uninstallError);
                            });

                        });

                        $(".hipchat-reinstall").click(function () {
                            if ($(this).attr('aria-disabled')) {
                                return;
                            }

                            var $el = $(this);
                            $el.attr('aria-disabled', true);
                            $el.find(".aui-icon-wait").css("display", "inline-block");

                            Remote.reinstall(linkId).done(Actions.redirect).error(Views.uninstallError);
                        });

                        if (localStorage.getItem("hipchat-reinstalling")) {
                            Views.reinstallationSucceeded();
                            localStorage.removeItem("hipchat-reinstalling");
                        }
                    },

                    uninstalling: function () {
                        Remote.removeDefault().done(function () {
                            Actions.redirect(routes.redirectUrl)
                        }).error(Views.uninstallError)
                    },

                    reinstalling: function () {
                        function getURLParameter(sParam) {
                            var sPageURL = window.location.search.substring(1);
                            var sURLVariables = sPageURL.split('&');
                            for (var i = 0; i < sURLVariables.length; i++)
                            {
                                var sParameterName = sURLVariables[i].split('=');
                                if (sParameterName[0] == sParam)
                                {
                                    if (sParameterName.length > 1) {
                                        return sParameterName[1];
                                    } else {
                                        return null;
                                    }
                                }
                            }
                        }
                        var paramValue = getURLParameter("installable_url");
                        if (paramValue === undefined) {
                            console.log("ERROR: the 'installable_url' parameter was missing from the request URL:'" + window.location.search + "'");
                        } else {
                            var url = decodeURIComponent(paramValue);
                            var a = document.createElement('a');
                            a.href = url;
                            var hipchatUrl = a.protocol + "//" + a.host;
                            Remote.serverPingToUrl(hipchatUrl)
                                    .done(function () {
                                        localStorage.setItem("hipchat-reinstalling", true);
                                        Remote.beginInstall(hipchatUrl, true).done(Actions.navigate)
                                    })
                                    .fail(function () {
                                        $("#btf-error").show();
                                    });
                        }
                    }

                };

                Actions.initialize(state);
            }
        }
    )

});
