define("jira-help-tips/feature/help-tip-manager", ["require"], function (require) {
    "use strict";

    var _ = require("underscore");
    var $ = require("jquery");
    var Deferred = require("jira/jquery/deferred");
    var contextPath = require("wrm/context-path");
    var data = require("wrm/data");

    var restUrl = contextPath() + "/rest/helptips/1.0/tips";
    var tipWRMData = data.claim("com.atlassian.plugins.helptips.jira-help-tips:help-tip-manager.JiraHelpTipData");

    var Manager = {
        dismissedTipIds: [],
        sequences: [],
        loaded: new Deferred(),
        url: function url() {
            return restUrl;
        },
        sync: function sync(verb, data) {
            var result = new Deferred();
            verb || (verb = "get");
            data || (data = null);
            if (verb === "get" && tipWRMData && tipWRMData.dismissed) {
                result.resolve(tipWRMData.dismissed);
            } else {
                $.ajax(this.url(), {
                    type: verb,
                    dataType: "json",
                    contentType: "application/json",
                    data: data && JSON.stringify(data),
                    processData: false
                }).done(function (data) {
                    result.resolve(data);
                }).fail(function () {
                    result.reject();
                });
            }
            return result.promise();
        },
        fetch: function fetch() {
            var result = this.sync();
            result.done($.proxy(function (response) {
                $.merge(this.dismissedTipIds, response);
                this.loaded.resolve();
            }, this));
            return result.promise();
        },
        show: function show(showFunction) {
            this.loaded.done(showFunction);
        },
        dismiss: function dismiss(tip) {
            var id = tip.attributes.id;
            if (!id) {
                tip._dismissed = true;
            } else {
                this.dismissedTipIds.push(id);
                this.sync("post", { id: id });
            }
        },
        undismiss: function undismiss(tip) {
            var id = tip.attributes.id;
            if (!id) {
                tip._dismissed = false;
            } else {
                this.dismissedTipIds.splice($.inArray(id, this.dismissedTipIds), 1);
                this.sync("delete", { id: id });
            }
        },
        isDismissed: function isDismissed(tip) {
            var id = tip.attributes.id;
            return id ? $.inArray(id, this.dismissedTipIds) >= 0 : tip._dismissed;
        },
        /**
         * Clear the list of helptips that would be shown in sequence
         */
        clearSequences: function clearSequences() {
            this.sequences = [];
        },
        /**
         * Hide all the helptips that would be shown in sequence
         */
        hideSequences: function hideSequences() {
            _.each(this.sequences, function (helptip) {
                helptip.view.dismiss();
            });
        },
        /**
         * Initiate the rendering of helptips in sequence starting from the first one
         */
        showSequences: function showSequences() {
            if (!this._showStarted) {
                var instance = this;
                var index = 0;
                this._showStarted = true;
                $.when(this.loaded).done(function () {
                    //Sort by weight
                    instance.sequences.sort(function (a, b) {
                        return a.attributes.weight - b.attributes.weight;
                    });
                    //Remove any helptips that have already been dismissed
                    instance.sequences = _.filter(instance.sequences, function (dialog) {
                        var shouldShow = _.indexOf(instance.dismissedTipIds, dialog.attributes.id) === -1;
                        if (shouldShow) {
                            dialog.attributes.position = index++;
                        }
                        return shouldShow;
                    });
                    if (instance.sequences.length > 0) {
                        //SW-1740: Ensure the last tip part of a sequence always shows the close button. Filtering out already dismissed
                        // tips above may have changed the original last tip.
                        _.last(instance.sequences).attributes.showCloseButton = true;

                        instance.sequences[0].show({ force: true });
                    }
                    instance._showStarted = false;
                });
            }
        }
    };

    return Manager;
});