define("jira-help-tips/feature/help-tip", ["require"], function (require) {
    "use strict";

    var $ = require("jquery");
    var _ = require("underscore");
    var Manager = require("jira-help-tips/feature/help-tip-manager");
    var analytics = require("jira-help-tips/util/analytics");
    var InlineDialog = require("aui/inline-dialog");
    var Deferred = require('jira/jquery/deferred');
    var Popups = AJS.Popups;
    var TEMPLATES = AJS.Templates.HelpTip;

    var cidCounter = 0;
    var seed = new Date().getTime();
    var HELP_TIP_CLASSNAMES = "jira-help-tip aui-help";

    function nope() {
        return false;
    }
    function yep() {
        return true;
    }

    function getCid() {
        return "jira-help-tip-" + seed + cidCounter++;
    }

    var AnchoredView = function AnchoredView(model, anchor) {
        this.initialize(model, anchor);
    };

    var UnanchoredView = function UnanchoredView(model) {
        this.initialize(model);
    };

    /**
     * @param attributes
     * @param attributes.id
     * @param attributes.callbacks callbacks used by helptip and passed into InlineDialog
     * @param attributes.callbacks.beforeShow {jQuery.Deferred() | Function} called before rendering the helptip
     * @param attributes.callbacks.init {Function} passed into InlineDialog as initCallback param
     * @param attributes.callbacks.hide {Function} passed into InlineDialog as hideCallback param
     * @param attributes.isSequence {Boolean} to indicate whether this helptip should be shown in sequence
     *        see https://developer.atlassian.com/design/latest/feature-discovery.html
     * @param attributes.weight {Integer} the weight of the helptip determining the order to be shown in sequence
     * @param attributes.body {String} the HTML body content of the helptip
     * @param attributes.bodyHtml {String} same as attributes.body
     * @param attributes.inlineDialogOpts {Map} passed into InlineDialog as options, This will replace all the defaults specified by HelpTip
     * @param attributes.eventPrefix {String} The prefix to use for all analytics events
     * @param attributes.nextButtonText {String} The text to be displayed on the 'next' button
     * @param attributes.closeButtonText {String} The text to be displayed on the 'close' button
     * @param attributes.showCloseButton {Boolean} Whether the 'close' button should be displayed on the helptip
     *
     * @type {Function}
     */
    var HelpTip = function HelpTip(attributes) {
        var anchor;
        this.attributes = $.extend({}, attributes);
        this.attributes.id || (this.attributes.id = false);
        this.attributes.callbacks || (this.attributes.callbacks = {});

        //Allows the helptip to be shown in sequence via next button
        if (this.attributes.isSequence) {
            if (!this.attributes.weight) {
                this.attributes.weight = Number.MAX_VALUE;
            }
            Manager.sequences.push(this);
        }

        // Map renamed attribute body to bodyHtml
        if (this.attributes.body) {
            this.attributes.bodyHtml = this.attributes.body;
            delete this.attributes.body;
        }
        this.cid = getCid();
        anchor = this.attributes['anchor'];
        delete this.attributes['anchor'];

        this.view = anchor ? new AnchoredView(this, anchor) : new UnanchoredView(this);
    };

    $.extend(HelpTip.prototype, {
        /**
         * @param options.force render the helptip regardless of the display controller
         */
        show: function show(options) {
            options = options || {};

            var self = this;
            var showDeferred = new Deferred();
            if (this.attributes.callbacks.beforeShow) {
                var beforeShowDeferred = this.attributes.callbacks.beforeShow();
                if (beforeShowDeferred && _.isFunction(beforeShowDeferred.done)) {
                    beforeShowDeferred.done(showDeferred.resolve);
                } else {
                    showDeferred.resolve();
                }
            } else {
                showDeferred.resolve();
            }

            showDeferred.done(function () {
                Manager.show(function () {
                    if (!self.isDismissed()) {
                        if (!options.force && Popups && Popups.DisplayController) {
                            Popups.DisplayController.request({
                                name: self.id,
                                weight: 1000,
                                show: function show() {
                                    self.view.show();
                                } });
                        } else {
                            self.view.show();
                        }
                        analytics.send("shown", self);
                    }
                });
            });
        },
        dismiss: function dismiss() {
            var reason = analytics.clean(arguments[0] || "programmatically");
            this.view.dismiss();
            //Clicking close should stop other helptip in sequence from showing
            if (reason === "close-button" && this.attributes.isSequence) {
                Manager.clearSequences();
            }
            if (!this.isDismissed()) {
                Manager.dismiss(this);
                analytics.send("dismissed." + reason, this);
            }
        },
        isVisible: function isVisible() {
            return this.view.$el.is(":visible");
        },
        isDismissed: function isDismissed() {
            return Manager.isDismissed(this);
        },

        /**
         * Redraws the help tip.
         *  - If a selector was passed as the anchor, it will check if the actual element has changed and reanchor
         *    if necessary
         *  - If the anchor is not visible, it will hide the help tip
         *  - If the anchor has moved, it will re-position the help tip
         */
        refresh: function refresh() {
            if (!this.isDismissed()) {
                this.view.refresh();
            }
        },

        /**
         * Hide the help tip without dismissing it
         */
        hide: function hide() {
            if (!this.isDismissed()) {
                this.view.dismiss();
            }
        },

        /**
         * Advance to the next help tip if this help tip
         * is part of a sequence and is not the last help
         * tip in the sequence. Useful in the situation where
         * user can progress by performing some action in addition to
         * clicking the next button.
         */
        showNextHelpTipInSequence: function showNextHelpTipInSequence() {
            this.view.clickNext();
        }
    });

    $.extend(AnchoredView.prototype, {
        initialize: function initialize(model, anchor) {
            this.model = model;
            this.anchorSelector = anchor;
            this.anchor = $(anchor);
            this._initDialog(anchor);
            $(document).bind("showLayer", function (e, type, layer) {
                if (type === "inlineDialog" && layer.id === model.cid) {
                    InlineDialog.current = null; // Tips shouldn't be considered InlineDialogs.
                    $(document.body).unbind("click." + model.cid + ".inline-dialog-check");
                    layer._validateClickToClose = nope;
                    layer.hide = nope;
                }
            });
        },
        show: function show() {
            this.beforeHide = nope;
            this.popup.show();
        },
        refresh: function refresh() {
            var maybeNewAnchor = $(this.anchorSelector);
            if (!maybeNewAnchor.is(':visible')) {
                this.dismiss(); // hide
            } else if (maybeNewAnchor.get(0) !== this.anchor.get(0)) {
                this.changeAnchor(maybeNewAnchor);
            } else if (!this.isVisible()) {
                this.show();
            } else {
                this.popup.refresh();
            }
        },
        changeAnchor: function changeAnchor(anchor) {
            var wasVisible = this.isVisible();
            this.dismiss();
            this.$el.remove();
            this.anchor = anchor;
            this._initDialog(anchor);
            if (wasVisible) {
                this.show();
            }
        },
        dismiss: function dismiss() {
            this.beforeHide = yep;
            this._popupHide();
        },
        clickNext: function clickNext() {
            var nextButton = $(this.$el).find(".helptip-next");
            if (nextButton.length > 0) {
                nextButton.click();
            }
        },
        isVisible: function isVisible() {
            return this.$el.is(':visible');
        },
        _initDialog: function _initDialog(anchor) {
            var self = this;
            var model = this.model;
            this.popup = new InlineDialog($(anchor), model.cid, _.bind(this._createDialog, this), _.extend({
                // Use a container other than body, so that the positioning works when there are client-rendered banner messages
                container: "#content",
                noBind: true,
                preHideCallback: function preHideCallback() {
                    return self.beforeHide();
                },
                calculatePositions: function calculatePositions(popup, targetPosition, mousePosition, opts) {
                    // Adjust positions relative to the container
                    var cssData = InlineDialog.opts.calculatePositions(popup, targetPosition, mousePosition, opts);
                    var $container = $(this.container);
                    var offset = $container.offset();
                    if (cssData.popupCss.left !== 'auto') {
                        cssData.popupCss.left -= offset.left;
                        // Inline dialog has a bug where in some scenarios both left and right pos of the popup is set.
                        cssData.popupCss.right = 'auto';
                    }
                    cssData.popupCss.top -= offset.top;
                    return cssData;
                },
                addActiveClass: false,
                initCallback: model.attributes.callbacks.init,
                hideCallback: model.attributes.callbacks.hide,
                persistent: true
            }, model.attributes.inlineDialogOpts));
            this._popupHide = this.popup.hide;
            this.popup.hide = nope;
            this.$el = $(this.popup[0]);
            this.$el.addClass(HELP_TIP_CLASSNAMES);
        },
        _createDialog: function _createDialog(content, trigger, show) {
            var instance = this;
            var sequenceDialogs = Manager.sequences;
            var position = this.model.attributes.position;
            var isSequence = this.model.attributes.isSequence;

            content.removeClass('contents');

            content.html($(TEMPLATES.tipContent(_.extend({
                showNextButton: isSequence && sequenceDialogs.length > 1 && position + 1 < sequenceDialogs.length,
                length: sequenceDialogs.length,
                position: position,
                showCloseButton: true
            }, this.model.attributes))));

            content.unbind('mouseover mouseout');
            content.find(".helptip-link").click(function () {
                analytics.send("learn-more.clicked", instance.model);
            });
            content.find(".helptip-close").click(function (e) {
                e.preventDefault();
                instance.model.dismiss("close-button");
            });
            content.find(".helptip-next").click(function (e) {
                e.preventDefault();
                instance.model.dismiss("next-button");
                var next = position + 1;
                sequenceDialogs[next] && sequenceDialogs[next].show({ force: true });
            });
            show();
        }
    });

    $.extend(UnanchoredView.prototype, {
        initialize: function initialize() {
            this.$el = $("<div></div>");
            this.$el.addClass(HELP_TIP_CLASSNAMES);
        },
        show: function show() {},
        dismiss: function dismiss() {}
    });

    return HelpTip;
});