define("jira-dashboard-items/components/filter-picker", ['jira/ajs/list/item-descriptor', 'jira/ajs/list/group-descriptor', 'wrm/context-path', 'underscore', 'jquery', 'jira/ajs/ajax/smart-ajax', 'jira/ajs/select/single-select', 'jira/analytics'], function (ItemDescriptor, GroupDescriptor, wrmContextPath, _, $, SmartAjax, SingleSelect, analytics) {

    'use strict';

    return function () {
        var observer;

        var getCurrentQuery = function getCurrentQuery(select) {
            return $("#" + select.model.$element.attr('id') + "-single-select").attr("data-query");
        };

        var queryEndsWithWildcard = function queryEndsWithWildcard(query) {
            if (_.isUndefined(query) || _.isEmpty(query)) {
                return false;
            } else {
                return ['?', '*', '~'].includes(query.charAt(query.length - 1));
            }
        };

        var getCurrentQueryState = function getCurrentQueryState(select) {
            var state = {
                EMPTY: "empty",
                REGULAR: "doesNotEndWithWildcard",
                WILDCARD: "endsWithWildcard"
            };

            var query = getCurrentQuery(select);

            if (_.isUndefined(query) || _.isEmpty(query)) {
                return state.EMPTY;
            } else if (queryEndsWithWildcard(query)) {
                return state.WILDCARD;
            } else {
                return state.REGULAR;
            }
        };

        var initialiseAdvancedSearch = function initialiseAdvancedSearch(parentElement, fieldId, select) {
            /*
             * Below MutationObserver listens for the updates from the advanced filter picker.
             */

            var updateSelect = function updateSelect() {
                var id = $("#filter_" + fieldId + "_id").attr("value");
                if (id.indexOf("filter-") === 0) {
                    id = id.substr(7);
                }
                var label = $("#filter_" + fieldId + "_name").text();
                var descriptor = new ItemDescriptor({
                    value: id,
                    label: label
                });

                select.setSelection(descriptor, true);
            };

            var advancedFilterResultIdField = parentElement.find("#filter_" + fieldId + "_id");

            observer = new MutationObserver(updateSelect);
            observer.observe(advancedFilterResultIdField[0], { attributes: true });

            parentElement.find("#filter_" + fieldId + "_advance").click(function (e) {
                var url = wrmContextPath() + "/secure/FilterPickerPopup.jspa?showProjects=false&field=" + fieldId;
                var windowVal = "filter_" + fieldId.replace(new RegExp('-', 'g'), '') + "_window";
                var prefs = "width=800, height=500, resizable, scrollbars=yes";

                var quickSearchInput = getCurrentQueryState(select);

                analytics.send({
                    name: 'jira.dashboard.gadgets.components.filter-picker.advanced-search.clicked',
                    data: {
                        quickSearchInput: quickSearchInput
                    }
                });

                window.open(url, windowVal, prefs).focus();

                e.preventDefault();
            });
        };

        return {
            init: function init(options) {
                var element = options.element;
                var selectedValue = options.selectedValue;
                var self = this;

                this.select = new SingleSelect({
                    element: element,
                    itemAttrDisplayed: "label",
                    revertOnInvalid: true,
                    removeDuplicates: true,
                    submitInputVal: false,
                    showDropdownButton: true,
                    removeOnUnSelect: false,
                    ajaxOptions: {
                        url: wrmContextPath() + "/rest/gadget/1.0/pickers/filters",
                        query: true,
                        data: {
                            fieldName: "quickfind"
                        },
                        formatResponse: function formatResponse(response) {
                            var ret = [];
                            if (response.filters && response.filters.length) {
                                var groupDescriptor = new GroupDescriptor();
                                _.each(response.filters, function (filter) {
                                    groupDescriptor.addItem(new ItemDescriptor({
                                        value: filter.id,
                                        label: filter.name
                                    }));
                                });
                                ret.push(groupDescriptor);
                            }
                            return ret;
                        }
                    }
                });

                this.select.model.$element.bind('selected', function (e, descriptor, select, event) {
                    if (_.isUndefined(event) || event.type === 'blur') {
                        return;
                    }

                    var query = getCurrentQuery(select);
                    if (!_.isUndefined(query)) {
                        var name;
                        var quickSearchInput = getCurrentQueryState(select);

                        if (select.lastSelection.label() === query) {
                            name = 'jira.dashboard.gadgets.components.filter-picker.exact-match.selected';
                        } else {
                            name = 'jira.dashboard.gadgets.components.filter-picker.suggestion.selected';
                        }

                        analytics.send({
                            name: name,
                            data: {
                                quickSearchInput: quickSearchInput
                            }
                        });
                    }
                });

                if (selectedValue) {
                    self.select.disable();

                    SmartAjax.makeRequest({
                        method: "GET",
                        url: wrmContextPath() + "/rest/api/2/filter/" + selectedValue,
                        success: function success(data) {
                            self.select.setSelection(new ItemDescriptor({
                                value: selectedValue,
                                label: data.name
                            }));
                        },
                        complete: function complete() {
                            self.select.enable();
                        }
                    });
                }

                initialiseAdvancedSearch(options.parentElement, element.attr('id'), self.select);

                this.errorField = element.parents(".field-group").find(".error");
                self.select.hideErrorMessage = function () {}; //noop
                self.select.showErrorMessage = function () {}; //noop
                return this;
            },

            getValue: function getValue() {
                if (this.select.lastSelection) {
                    return {
                        "id": this.select.lastSelection.value(),
                        "label": this.select.lastSelection.label()
                    };
                }
                return null;
            },

            validate: function validate() {
                if (this.select.lastSelection && this.select.lastSelection.value()) {
                    this.errorField.hide();
                    return true;
                } else {
                    this.errorField.show();
                    return false;
                }
            },

            disconnectObserver: function disconnectObserver() {
                if (observer) {
                    observer.disconnect();
                }
            }
        };
    };
});