define("jira/components/testutils/app-module", ["require"], function(require) {
    "use strict";

    var _ = require("jira/components/libs/underscore");
    var Application = require('jira/components/libs/marionette-1.4.1/application');
    var AJSTest = AJS.test;
    var globalSinon = window.sinon;

    return {

        /**
         * Creates a new JIRA.Marionette.Application and starts it.
         *
         * @returns {JIRA.Marionette.Application}
         */
        startApplication: function(options) {
            var application = new Application();
            application.start(options);
            return application;
        },

        /**
         * Creates an AppModule using an AMD definition and installs it into an application.
         *
         * This method returns the instance of Module that was created by the AppModule, which can
         * be used for testing.
         *
         * @param {Object} options - Wrapper for the options of this method
         * @param {string} options.appModuleName - AMD name of the AppModule to install
         * @param {string} options.moduleName - AMD name of the Module
         * @param {JIRA.Marionette.Application} [options.application] - Application where this appModule should be
         *        installed. If not provided, this method will create a new application using `startApplication()`.
         *        In that case, the created application is not returned, so if you want to do further asserts in the
         *        application, you have to create your own.
         * @param {Object} [options.sinon=window.sinon] - Sinon instance to use for creating the spies/stubs. By default
         *        it uses `window.sinon`. This option also accepts a sinon sandbox, which can be used lately to restore
         *        all the spies/stubs at once.
         * @param {string} [options.mockableModuleContext] - Instance of AJS.test.mockableModuleContext to use for mocks.
         *        By default it creates a new instance.
         */
        createAMDModule: function(options) {
            var moduleName = options.moduleName;
            if (!moduleName) throw new Error("Missing options.moduleName");

            var appModuleName = options.appModuleName;
            if (!appModuleName) throw new Error("Missing options.appModuleName");

            var mockableModuleContext = options.mockableModuleContext || AJSTest.mockableModuleContext();
            var sandbox = options.sinon || globalSinon;

            // Load the real module and spy on all the methods
            var module = require(moduleName);
            var moduleSpy = sandbox.spy(module);

            // Inject the spied version to the mocked context
            mockableModuleContext.mock(moduleName, moduleSpy);

            // Load the AppModule from the mocked context (this will use the spied Module)
            var AppModule = mockableModuleContext.require(appModuleName);

            // Install it in the JIRA.Marionette.Application
            var application = options.application || this.startApplication();
            application.module(moduleName, new AppModule().definition);

            // Return the *instance* of Module that is actually running in the application
            return moduleSpy.firstCall.thisValue;
        },

        /**
         * Creates a stub for each method in the list.
         *
         * @param {Object} options
         * @param {string[]} options.methods - List of methods to stub.
         * @param {Function} options.moduleClass - Main class where methods reside. The methods will be stub from the class' prototype.
         * @param {Object} [options.sinon=window.sinon] - Sinon instance (or sandbox) to use for creating the stubs.
         */
        stubMethods: function(options) {
            var methods = options.methods;
            if (!methods) throw new Error("Missing options.methods");

            var moduleClass = options.moduleClass;
            if (!moduleClass) throw new Error("Missing options.moduleClass");

            var sandbox = options.sinon || globalSinon;

            _.each(methods, function(method) {
                if (!(method in moduleClass.prototype)) {
                    throw new Error("Method '" + method + "' not found in the module");
                }
                sandbox.stub(moduleClass.prototype, method);
            });
        },

        /**
         * Asserts that a list of commands are implemented by the application, and they call the method with the same name
         * in the internal object created by the AppModule.
         *
         * @param {Object} options
         * @param {string[]} options.commands - Commands to check.
         * @param {string} options.commandsNamespace - Namespace used for the commands (a command is in the form <namespace>:<command>).
         * @param {JIRA.Marionette.Application} options.application - Application where the commands have been installed.
         * @param {Object} options.module - Object that implements the handlers for the commands. These methods must be a spy/stub.
         */
        assertCommands: function(options) {
            var commands = options.commands;
            if (!commands) throw new Error("Missing options.commands");

            var commandsNamespace = options.commandsNamespace;
            if (!commandsNamespace) throw new Error("Missing options.commandsNamespace");

            var application = options.application;
            if (!application) throw new Error("Missing options.application");

            var module = options.module;
            if (!module) throw new Error("Missing options.module");

            _.each(commands, function(command) {
                application.execute(commandsNamespace + ":" + command);
                ok(module[command].calledOnce, "The command '" + command + "' calls the method with the same name");
            });
        },

        /**
         * Asserts that a list of requests are implemented by the application, and they call the method with the same name
         * in the internal object created by the AppModule.
         *
         * @param {Object} options
         * @param {string[]} options.requests Requests to check
         * @param {string} options.requestsNamespace Namespace used for the requests (a request is in the form <namespace>:<request>)
         * @param {JIRA.Marionette.Application} options.application Application where the request have been installed
         * @param {Object} options.module Object that implements the handlers for the request. These methods must be a spy/stub
         */
        assertRequests: function(options) {
            var request = options.requests;
            if (!request) throw new Error("Missing options.request");

            var requestNamespace = options.requestsNamespace;
            if (!requestNamespace) throw new Error("Missing options.requestNamespace");

            var application = options.application;
            if (!application) throw new Error("Missing options.application");

            var module = options.module;
            if (!module) throw new Error("Missing options.module");

            _.each(request, function(request) {
                application.request(requestNamespace + ":" + request);
                ok(module[request].calledOnce, "The request '" + request + "' calls the method with the same name");
            });
        },

        /**
         * Asserts that a list of events fire an event in the Application when the event with the same name is fired
         * in the internal object created by the AppModule.
         *
         * @param {Object} options
         * @param {string[]} options.events - Events to check.
         * @param {Object} [options.sinon=window.sinon] - Sinon instance (or sandbox) to use for creating the stubs.
         * @param {JIRA.Marionette.Application} options.application - Application where the commands have been installed.
         * @param {string} options.eventsNamespace - Namespace used for the events (a event is in the form <namespace>:<event>).
         * @param {Object} options.module - Object that will fire the events.
         */
        assertEvents: function(options) {
            var events = options.events;
            if (!events) throw new Error("Missing options.events");

            var sandbox = options.sinon || globalSinon;

            var application = options.application;
            if (!application) throw new Error("Missing options.application");

            var eventsNamespace = options.eventsNamespace;
            if (!eventsNamespace) throw new Error("Missing options.eventsNamespace");

            var module = options.module;
            if (!module) throw new Error("Missing options.module");

            _.each(events, function(event) {
                var spy = sandbox.spy();
                application.on(eventsNamespace + ":" + event, spy);
                module.trigger(event);
                ok(spy.calledOnce, "The event '" + event + "' was triggered in the application");
            });
        }
    };
});
