define("jira/components/search/results", ['require'], function (require) {
    "use strict";

    var wrmContextPath = require('wrm/context-path');
    var PageableCollection = require('jira/components/libs/backbone.paginator');
    var _ = require('jira/components/libs/underscore');
    var jQuery = require('jquery');
    var Result = require('jira/components/search/result');
    var Deprecator = require("jira/legacy/deprecator");

    var Results = PageableCollection.extend({
        initialize: function (models, options) {
            options = options || {};
            this.allIssues = options.issues;
            this.state.pageSize = options.pageSize;
            this.state.totalRecords = options.totalRecordsInSearch;
            this.totalRecordsInDB = options.totalRecordsInDB;
            this.allowNoSelection = options.allowNoSelection;
            this.jql = options.jql;
        },

        model: Result,

        url: function () {
            return wrmContextPath() + "/rest/issueNav/1/issueTable/stable";
        },

        // Initial pagination states
        state: {
            firstPage: 0,
            currentPage: 0
        },

        queryParams: {
            currentPage: null,
            pageSize: null,
            totalPages: null,
            totalRecords: null,
            sortKey: null,
            directions: null
        },

        sync: function () {
            var args = _.toArray(arguments);
            var params = args[2];

            params.type = "POST";
            params.headers = {
                "X-Atlassian-Token": "no-check"
            };
            params.data.layoutKey = "split-view";
            params.data.id = this._getStableKeys();

            return PageableCollection.prototype.sync.apply(this, args);
        },

        isAtTheEndOfStableSearch: function () {
            var isLastPage = this.state.currentPage === this.state.lastPage;
            var areThereMoreIssues = this.totalRecordsInDB > this.state.totalRecords;
            return isLastPage && areThereMoreIssues;
        },

        /**
         * @param preloadedData expects an object like this: { ids: [1000,1001,10002], response: {..ajax response..}}
         */
        setPreloadedData: function(preloadedData) {
            this.preloadedData = preloadedData;
            this.reset(preloadedData.response, { parse: true });
        },

        _getStableKeys: function (startIndex) {
            var start = typeof startIndex === 'undefined' ? this._getStartIndex() : startIndex;
            return _.pluck(this.allIssues, "id").slice(start, start + this.state.pageSize);
        },

        _getStartIndex: function () {
            return this.state.pageSize * this.state.currentPage;
        },

        _loadPageAndSelect: function (pageToLoad, issueToSelect) {
            var deferred = new jQuery.Deferred();

            var options = {fetch: true, reset: true};
            this.trigger("before:loadpage", {
                pageToLoad: pageToLoad
            });
            this.getPage(pageToLoad, options)
                .done(_.bind(function () {
                    this.select(issueToSelect);
                    deferred.resolve(this.length);
                }, this))
                .fail(_.bind(function(response) {
                    deferred.reject(response);
                    this.trigger("error:loadpage", response);
                }, this));

            return deferred.promise();
        },

        fetch: function(options) {
            var promise;
            if (this.preloadedData && _.isEqual(this.preloadedData.ids, this._getStableKeys(this.state.pageSize * options.to))) {
                this.reset(this.preloadedData.response, { parse: true });
                promise = new jQuery.Deferred().resolve().promise();
                delete this.preloadedData;
            } else {
                promise = PageableCollection.prototype.fetch.call(this, options);
            }
            return promise;
        },

        jumpToPage: function (page) {
            var deferred = new jQuery.Deferred();
            this._loadPageAndSelect(page, "first")
                .done(deferred.resolve)
                .fail(deferred.reject);
            return deferred.promise();
        },

        jumpToPageForIssue: function (issueIdOrKey) {
            var deferred = new jQuery.Deferred();
            var isIssueId = (typeof issueIdOrKey === "number");
            var isIssueKey = (typeof issueIdOrKey === "string");

            // Find which page the issue belongs to
            var pageToLoad;
            var issue;
            for (var i = 0, len = this.allIssues.length; i < len && !issue; i++) {
                if (
                    (isIssueId && this.allIssues[i].id === issueIdOrKey) ||
                    (isIssueKey && this.allIssues[i].key === issueIdOrKey)
                ) {
                    issue = this.allIssues[i];
                    pageToLoad = Math.floor(i / this.state.pageSize);
                }
            }

            if (typeof pageToLoad === "number" && isFinite(pageToLoad)) {
                // Page for issue found, go to the page and select the issue
                this._loadPageAndSelect(pageToLoad, issue.id).done(deferred.resolve);
            } else if (this.allowNoSelection) {
                // Page for issue not found, but we allow having no selection
                // Load the page but do not select anything
                this._loadPageAndSelect("first", null).done(_.bind(function(){
                    this.trigger("selectIssueNotInList", new Result({
                        id: isIssueId ? issueIdOrKey : undefined,
                        key: isIssueKey ? issueIdOrKey : undefined
                    }));
                }, this)).done(deferred.resolve);
            } else {
                // Page for issue not found and we require a selection, select
                // the first issue in the first page
                this._loadPageAndSelect("first", "first").done(deferred.resolve);
            }
            return deferred.promise();
        },

        parse: function (resp, options) {
            var requestData = options.data;
            var data = resp && resp.issueTable;

            var allIssues = this.allIssues;
            _.each(data.table, function (rawIssue, index) {
                // If an issue is missing from the server, it will contain the value 'null'
                if (rawIssue === null) {
                    // We take advantage of the fact that the order of requestData and the response is the same.
                    var issueId = requestData.id[index];

                    // Search for this particular issue in the list of issues for this search.
                    var oldIssueData = _.findWhere(allIssues, {id: issueId});
                    if (oldIssueData) {
                        // Even if the issue is missing from the server, populate the required fields (id and key)
                        // so all our views, events and url management will continue to work.
                        data.table[index] = {
                            id: oldIssueData.id,
                            key: oldIssueData.key,
                            inaccessible: true
                        };
                    }
                }
            });

            return PageableCollection.prototype.parse.call(this, data);
        },

        parseState: function () {
            return {};
        },

        parseRecords: function (resp) {
            if (resp.table) {
                return resp.table;
            }
            else {
                return [];
            }
        },

        select: function (model) {
            if (typeof model === "string") {
                switch (model) {
                    case "first":
                        model = this.at(0);
                        break;
                    case "last":
                        model = this.at(this.length - 1);
                        break;
                    default:
                        model = this.findWhere({key: model});
                        break;
                }
            } else if (typeof model === "number") {
                model = this.get(model);
            }

            if (this.selected !== model) {
                this.unselect();
            }
            if (!model) {
                return;
            }
            this.selected = model;
            this.selected.trigger("select", this.selected);
        },

        unselect: function () {
            if (this.selected) {
                this.selected.trigger("unselect", this.selected);
            }
            this.selected = null;
        },

        navigate: function (step) {
            var index = this.indexOf(this.selected);
            var newIndex = index + step;
            var model = this.at(newIndex);
            if (model) {
                this.select(model);
            }
            else if (step > 0 && this.hasNextPage()) {
                this._loadPageAndSelect("next", "first");
            } else if (step < 0 && this.hasPreviousPage()) {
                this._loadPageAndSelect("prev", "last");
            }
            return {current: newIndex, total: this.length};
        },

        selectNext: function () {
            return this.navigate(1);
        },

        selectPrev: function () {
            return this.navigate(-1);
        },

        /**
         * Gets the issue based on an offset from the current selected issue.
         * @param {number} offset Offset (eg: '1' gets the next issue, '-1' gets the previous issue)
         * @returns {jQuery.Promise} Promise that will be resolved with the selected issue, or rejected if the issue
         *          does not exist.
         * @private
         */
        _getIssueByNavigation: function(offset) {
            var index = this.indexOf(this.selected);
            var newIndex = index + offset;
            var model = this.at(newIndex);
            var deferred = new jQuery.Deferred();

            if (model) {
                this.select(model);
                deferred.resolve(model);
            } else if (offset > 0 && this.hasNextPage()) {
                this._loadPageAndSelect("next", "first").done(function() {
                    deferred.resolve(this.selected);
                }.bind(this));
            } else if (offset < 0 && this.hasPreviousPage()) {
                this._loadPageAndSelect("prev", "last").done(function() {
                    deferred.resolve(this.selected);
                }.bind(this));
            } else {
                deferred.reject();
            }

            return deferred.promise();
        },

        /**
         * Gets the next issue
         *
         * @returns {jQuery.Promise} Promise that will be resolved with the selected issue, or rejected if the issue
         *          does not exist.
         */
        getNextIssue: function() {
            return this._getIssueByNavigation(1);
        },

        /**
         * Gets the previous issue
         *
         * @returns {jQuery.Promise} Promise that will be resolved with the selected issue, or rejected if the issue
         *          does not exist.
         */
        getPreviousIssue: function() {
            return this._getIssueByNavigation(-1);
        },

        updateSelectedIssue: function () {
            if (this.selected) {
                this.selected.fetch();
            }
        },

        updateIssue: function(issueId) {
            var issue = this.get(issueId);
            if (issue) {
                issue.fetch();
            }
        },

        hasIssueInSearch: function (issueKey) {
            return !!_.where(this.allIssues, {key: issueKey}).length;
        },

        isEmptySearch: function () {
            return !this.allIssues.length;
        },

        /**
         * This method removes the model from the collection. If the model was selected, it will
         * try to select a new issue based on the following algorithm:
         *
         *  * If there is a next issue, select it. ...
         *  * If there is a previous issue, select it. Else...
         *  * The list is empty, select nothing
         *
         * @param {JIRA.Components.Search.Result} model Issue to delete
         */
        removeAndUpdateSelectionIfNeeded: function (model) {
            var issueKey = model.get('key');
            var issueToSelectAfterRemoveOperation;
            var allKeys = _.chain(this.allIssues).map(function (issue) {return issue.key;}).value();
            var currentIndex = _.indexOf(allKeys, issueKey);
            var hasNext = currentIndex + 1 < allKeys.length;
            var hasPrevious = currentIndex - 1 >= 0;

            if (hasNext) {
                issueToSelectAfterRemoveOperation = allKeys[currentIndex + 1];
            }
            else if (hasPrevious) {
                issueToSelectAfterRemoveOperation = allKeys[currentIndex - 1];
            }
            else {
                // There are no more issues, select nothing
                issueToSelectAfterRemoveOperation = null;
            }

            // Remove the issue from this.allIssues
            for (var i = 0; i < this.allIssues.length; i++) {
                var issue = this.allIssues[i];
                if (issue.key === issueKey) {
                    this.allIssues.splice(i, 1);
                    break;
                }
            }

            // Remove the issue from the pagination related counters
            this.state.totalRecords = this.state.totalRecords - 1;
            this.totalRecordsInDB = this.totalRecordsInDB - 1;

            // Unselect and remove it
            this.unselect();
            this.remove(model);

            // If we have an issue to select after the remove operation, load its page
            if (issueToSelectAfterRemoveOperation) {
                return this.jumpToPageForIssue(issueToSelectAfterRemoveOperation).done(_.bind(function() {
                    this.trigger("issueDeleted");
                }, this));
            } else {
                return new jQuery.Deferred().resolve(0).promise().done(_.bind(function() {
                    this.trigger("issueDeleted");
                }, this));
            }
        },

        getIssueKeyForIndex: function(index) {
            var issue = this.allIssues[index];
            if(!issue) return;

            return issue.key;
        },

        getIssueAtGlobalIndex: function(index) {
            return this.allIssues[index];
        },


        getPositionOfIssueInSearchResults: function(issueId) {
            var position = _.indexOf(_.pluck(this.allIssues, 'id'), issueId);
            if (position > -1) {
                return position;
            } else {
                return null;
            }
        },

        getPositionOfIssueInPage: function(issueId) {
            var model = this.get(issueId);
            var position = this.indexOf(model);
            if (position > -1) {
                return position;
            } else {
                return null;
            }
        },

        /**
         * @deprecated In JIRA 7.0. Will be removed in JIRA 8.0. Use getTotalIssuesInSearch() instead;
         */
        getTotalIssues: function() {
            return this.getTotalIssuesInSearch();
        },

        getTotalIssuesInSearch: function() {
            return this.allIssues.length;
        },

        getTotalIssuesInDb: function() {
            return this.totalRecordsInDB;
        }
    });

    Deprecator.prop(Results.prototype, 'getTotalIssues', {
        removeInVersion: '8.0',
        alternativeName: 'jira/components/search/results#getTotalIssuesInSearch()',
        sinceVersion: '7.1',
        displayName: "Method jira/components/search/results#getTotalIssues()"
    });

    return Results;
});
