define("jira/components/search", ['require'], function (require) {
    "use strict";

    var wrmContextPath = require('wrm/context-path');
    var Backbone = require('jira/components/libs/backbone');
    var _ = require('jira/components/libs/underscore');
    var jQuery = require('jquery');
    var Results = require('jira/components/search/results');
    var LatestSearchStore = require('jira/components/search/latestStorage');

    /**
     * This class contains the model for a Search. A Search is a collection of issues as a result
     * of running a JQL in the server. This entity is not meant to be used directly by any view.
     * Instead, this entity will generate a SearchResults entity that contains the stable search.
     *
     * All operations performed in this entity will result in a new set of results
     * (aka un-stable search).
     */
    return Backbone.Collection.extend({
        url: function () {
            return wrmContextPath() + "/rest/issueNav/1/issueTable";
        },

        searchParams: {
            jql: "",
            filterId: null,
            columnConfig: "explicit",
            columns: [],
            layoutKey: "split-view",
            startIndex: 0
        },

        sync: function () {
            var args = _.toArray(arguments);
            var params = args[2];

            params.type = "POST";
            params.headers = {
                "X-Atlassian-Token": "no-check"
            };

            // Add our search data.
            params.data = {};
            _.each(_.keys(this.searchParams), function (key) {
                if (this.searchParams[key] !== null) {
                    params.data[key] = this.searchParams[key];
                }
            }, this);

            return Backbone.Collection.prototype.sync.apply(this, args);
        },

        parse: function (resp) {
            // Clean the data
            var data = resp && resp.issueTable;
            if (!data || !data.issueIds) {
                return [];
            }

            // Extract all the issues from the first search.
            return _.map(data.issueIds, function (id, index) { return {id: id, key: data.issueKeys[index]}; });
        },

        update: function(response) {
            this.set(this.parse(response));
        },

        /**
         * @param preloadedData expects an object like { jql: "..jql used..", response: {..ajax response..} }
         */
        setPreloadedData: function(preloadedData) {
            this.preloadedData = preloadedData;
            this.reset(preloadedData.response, { parse: true });
            return this.parseResults(preloadedData.jql, preloadedData.response);
        },

        parseResults: function(jql, response) {
            LatestSearchStore.save({jql: jql});
            return new Results([], {
                issues: this.toJSON(),
                pageSize: response.issueTable.pageSize,
                totalRecordsInDB: response.issueTable.total,
                totalRecordsInSearch: this.length,
                jql: jql
            });
        },

        search: function (jql) {
            this.searchParams.jql = jql;

            var deferred = new jQuery.Deferred();

            if (this.preloadedData && this.preloadedData.jql === jql) {
                this.reset(this.preloadedData.response, { parse: true });
                deferred.resolve(this.parseResults(jql, this.preloadedData.response));
                delete this.preloadedData;
            } else {
                this.fetch({
                    reset: true,
                    success: _.bind(function (collection, response) {
                        deferred.resolve(this.parseResults(collection.searchParams.jql, response));
                    }, this),
                    error: function (collection, response) {
                        deferred.reject(response);
                    }
                });
            }

            return deferred.promise();
        }
    });
});
