define("jira/components/query/querystatemodel", ["require"], function(require) {
    "use strict";

    var Brace = require("jira/components/libs/brace");
    var Messages = require("jira/components/query/messages");
    var wrmContextPath = require('wrm/context-path');
    var jQuery = require("jquery");
    var trace = require('jira/components/utils/trace');


    /**
     * Represents current and preferred search modes (basic or jql)
     */
    return Brace.Model.extend({

        BASIC_SEARCH: "basic",
        ADVANCED_SEARCH: "advanced",

        namedAttributes: [
            "style",
            "searchMode",
            "preferredSearchMode",
            "jql",
            "context",
            "without",
            "layoutSwitcher",
            "autocompleteEnabled",
            "advancedAutoUpdate",
            "basicAutoUpdate",
            "basicOrderBy"
        ],

        defaults: {
            searchMode: "basic",
            preferredSearchMode: "basic"
        },

        /**
         * Sets search mode
         * @param searchMode search mode (basic or advanced)
         */
        switchToSearchMode: function(searchMode) {
            this.setSearchMode(searchMode);
        },

        /**
         * Changes the preferred and actual search mode and saves the preferred search mode.
         */
        switchPreferredSearchMode: function(mode) {
            this.switchToSearchMode(mode);
            this.setPreferredSearchMode(mode);
            this._savePreferredSearchMode();
        },

        /**
         * Switches to whatever is the preferred search mode
         */
        switchToPreferredSearchMode: function() {
            this.switchToSearchMode(this.getPreferredSearchMode());
        },

        hasSearchButton: function() {
            return this.getStyle() !== "field";
        },

        /**
         * Should the more criteria button be subtly styled
         */
        hasSubtleMoreCriteria: function() {
            return this.getStyle() !== "field";
        },

        /**
         * combine the jql with the context jql. The purpose of this method is narrow down the option values in dropdownlist following the context
         * (for ex: in project, if project does not define issue type "Bug", the dropdown list should not render "Bug" option)
         * contextJql (for ex: project = "Service Desk")
         * @return string: a complete jql in a scope. (project = "Service Desk' AND issuetype = "IT Help")
         */
        getJqlWithContext: function() {
            return this.combineContextAndJql(this.getContext(), this.getJql());
        },

        /**
         * Take the input jql and prepend the jql context
         * Does not mutate the state of the Model
         * @param context string (project = "Service Desk")
         * @param jql string (issuetype = "IT Help")
         * @return string: a complete jql in a scope. (project = "Service Desk' AND issuetype = "IT Help")
         */
        combineContextAndJql: function(context, jql) {
            // doing a filter and join takes care of the case where either is empty
            return [context, jql].filter(Boolean).join(" AND ");
        },

        /**
         * Persists preferred search mode to the server
         */
        _savePreferredSearchMode: function() {
            jQuery.ajax({
                url: wrmContextPath() + "/rest/querycomponent/latest/userSearchMode", // IssueTableResource (JIRA core)
                type: 'POST',
                headers: {
                    "X-Atlassian-Token": "no-check"
                },
                data: {
                    searchMode: this.getPreferredSearchMode()
                },
                error: function(xhr) {
                    Messages.displayFailSearchMessage(xhr);
                },
                success: function() {
                    trace("jira.search.mode.changed");
                }
            });
        }
    });
});
