define("jira/components/issueeditor/inlineeditutils", ["require"], function(require){
    "use strict";

    var Events = require('jira/util/events');
    var EventTypes = require("jira/components/issueeditor/eventtypes");

    /**
     * The time required to wait between blur and focus
     */
    var InlineEditUtils = {};

    InlineEditUtils.BLUR_FOCUS_TIMEOUT = 50;

    InlineEditUtils.BlurTriggers = {
        Default: function(fieldId, $container) {

            // Include save buttons
            $container = $container.nextAll(".save-options").andSelf();

            var focusables = ':input, a[href], [tabindex]';
            var timeout;
            var containerHasFocus = hasFocus($container);
            var eventsMap = {
                focus: function() {
                    if (!containerHasFocus) {
                        containerHasFocus = true;
                        Events.trigger(EventTypes.INLINE_EDIT_FOCUSED, [fieldId]);
                    }
                },
                blur: function() {
                    if (timeout) clearTimeout(timeout);
                    timeout = setTimeout(triggerIfBlurred, InlineEditUtils.BLUR_FOCUS_TIMEOUT);
                }
            };

            // Make container focusable
            $container.attr('tabindex', 1)
                // Bind to container
                .bind(eventsMap)
                // Bind to focusable elements in container
                .delegate(focusables, eventsMap);

            function triggerIfBlurred() {
                if (!hasFocus($container)) {
                    containerHasFocus = false;
                    Events.trigger(EventTypes.INLINE_EDIT_BLURRED, [fieldId]);
                }
            }
        }
    };

    function hasFocus($container) {
        var activeElement = document.activeElement;
        return $container.find(activeElement).length > 0 || $container.filter(activeElement).length > 0;
    }

    InlineEditUtils.BlurTriggerMapping = {
        system: {
            "summary": InlineEditUtils.BlurTriggers.Default,
            "priority": InlineEditUtils.BlurTriggers.Default,
            "issuetype": InlineEditUtils.BlurTriggers.Default,
            "components": InlineEditUtils.BlurTriggers.Default,
            "versions": InlineEditUtils.BlurTriggers.Default,
            "fixVersions": InlineEditUtils.BlurTriggers.Default,
            "assignee": InlineEditUtils.BlurTriggers.Default,
            "reporter": InlineEditUtils.BlurTriggers.Default,
            "environment": InlineEditUtils.BlurTriggers.Default,
            "description": InlineEditUtils.BlurTriggers.Default,
            "labels": InlineEditUtils.BlurTriggers.Default,
            "duedate": InlineEditUtils.BlurTriggers.Default
        },
        custom: {
            "com.atlassian.jira.plugin.system.customfieldtypes:cascadingselect": InlineEditUtils.BlurTriggers.Default,
            "com.atlassian.jira.plugin.system.customfieldtypes:datepicker": InlineEditUtils.BlurTriggers.Default,
            "com.atlassian.jira.plugin.system.customfieldtypes:datetime": InlineEditUtils.BlurTriggers.Default,
            "com.atlassian.jira.plugin.system.customfieldtypes:float": InlineEditUtils.BlurTriggers.Default,
            "com.atlassian.jira.plugin.system.customfieldtypes:grouppicker": InlineEditUtils.BlurTriggers.Default,
            "com.atlassian.jira.plugin.system.customfieldtypes:labels": InlineEditUtils.BlurTriggers.Default,
            "com.atlassian.jira.plugin.system.customfieldtypes:multicheckboxes": InlineEditUtils.BlurTriggers.Default,
            "com.atlassian.jira.plugin.system.customfieldtypes:multigrouppicker": InlineEditUtils.BlurTriggers.Default,
            "com.atlassian.jira.plugin.system.customfieldtypes:multiselect": InlineEditUtils.BlurTriggers.Default,
            "com.atlassian.jira.plugin.system.customfieldtypes:multiuserpicker": InlineEditUtils.BlurTriggers.Default,
            "com.atlassian.jira.plugin.system.customfieldtypes:multiversion": InlineEditUtils.BlurTriggers.Default,
            "com.atlassian.jira.plugin.system.customfieldtypes:project": InlineEditUtils.BlurTriggers.Default,
            "com.atlassian.jira.plugin.system.customfieldtypes:select": InlineEditUtils.BlurTriggers.Default,
            "com.atlassian.jira.plugin.system.customfieldtypes:radiobuttons": InlineEditUtils.BlurTriggers.Default,
            "com.atlassian.jira.plugin.system.customfieldtypes:textarea": InlineEditUtils.BlurTriggers.Default,
            "com.atlassian.jira.plugin.system.customfieldtypes:textfield": InlineEditUtils.BlurTriggers.Default,
            "com.atlassian.jira.plugin.system.customfieldtypes:url": InlineEditUtils.BlurTriggers.Default,
            "com.atlassian.jira.plugin.system.customfieldtypes:userpicker": InlineEditUtils.BlurTriggers.Default,
            "com.atlassian.jira.plugin.system.customfieldtypes:version": InlineEditUtils.BlurTriggers.Default
        }
    };

    Events.bind(EventTypes.INLINE_EDIT_STARTED, function(e, fieldId, fieldType, $container) {
        var blurTrigger;
        if (fieldType) {
            blurTrigger = InlineEditUtils.BlurTriggerMapping.custom[fieldType];
            if (!blurTrigger) {
                // in 5.1, we told devs to put the "short" key into the custom mapping, not the complete key
                // if we can't find the complete key, try the short key
                var shortKey = fieldType.replace(/.+:(.+)/, "$1"); // extract after the colon
                blurTrigger = InlineEditUtils.BlurTriggerMapping.custom[shortKey];
            }
        } else {
            blurTrigger = InlineEditUtils.BlurTriggerMapping.system[fieldId];
        }

        if (blurTrigger) {
            blurTrigger(fieldId, $container);
        }
    });

    return InlineEditUtils;
});
