define('wiki-edit/WikiEnabledTextarea', [
    'jira/util/formatter',
    'underscore',
    'wiki-edit/TextareaManipulator',
    'wiki-edit/UndoableTextarea',
    'wiki-edit/keymaster',
    'wiki-edit/keymaster-setup',
    'wiki-edit/SpeechRecognition',
    'jquery'
], function(
    formatter,
    _,
    TextareaManipulator,
    UndoableTextarea,
    onKey,
    keymasterSetup,
    speechRecognition,
    $) {
    keymasterSetup.init();

    var editorCount = 0;
    // matches any combination of "*" and "#" followed by space which is any list, no matter how deeply nested
    var autocontinuingPrefixMarkupRegexp = /^[#\*]+ /;

    var WikiEnabledTextarea = function(element) {
        this.el = element;
        this.$el = $(element);
        if (this.$el.data("wikiEditor")) {
            // this will be very rare so defensive caching of this value doesn't seem to make much sense
            return this.$el.data("wikiEditor");
        }

        this.id = "wikiEdit" + editorCount;
        editorCount += 1;
        this.manipulationEngine = new TextareaManipulator(this.el);
        this.undoRedoEl = new UndoableTextarea(element);

        var wikiEditor = this;
        // on enter, continue list if possible or discontinue it if current list item is empty
        onKey("enter", wikiEditor.id, function onEnterKey(e) {
            var selection = wikiEditor.manipulationEngine.getSelection();
            var prefix = wikiEditor.manipulationEngine.getFirstLineMatch(autocontinuingPrefixMarkupRegexp);
            if (prefix) {
                if (selection.length == 0 && wikiEditor.manipulationEngine.getLineAtCursor() === prefix) {
                    wikiEditor.manipulationEngine.unprefixSelectedLines(prefix);
                } else {
                    wikiEditor.manipulationEngine.replaceSelectionWith("\n" + prefix);
                }
                e.preventDefault();
            }
        });

        this.$el.data("wikiEditor", this);
        this.$el.addClass("wiki-editor-initialised");
    };

    /**
     * Standard wiki markup in the form of <prefix>text<suffix> like *bold*, _italic_ or [mailto:someone@example.com].
     * Every time such markup is inserted:
     *     - if user didn't select any text, the placeholder text will be inserted and selected,
     *     - if user selected some text, this text will be wrapped with prefix and suffix.
     *
     * Each of this properties gets a function on the prototype with the same name
     * that changes value of textarea to apply the markup.
     */
    var wrapperMarkup = {
        'bold': ["*", "*", formatter.I18n.getText("jira.wiki.editor.operation.bold.placeholder")],
        'italic': ["_", "_", formatter.I18n.getText("jira.wiki.editor.operation.italic.placeholder")],
        'underline': ["+", "+", formatter.I18n.getText("jira.wiki.editor.operation.underline.placeholder")],
        'delete': ["-", "-", formatter.I18n.getText("jira.wiki.editor.operation.strikethrough.placeholder")],
        'superscript': ["^", "^", formatter.I18n.getText("jira.wiki.editor.operation.superscript.placeholder")],
        'subscript': ["~", "~", formatter.I18n.getText("jira.wiki.editor.operation.subscript.placeholder")],
        'monospace': ["{{", "}}", formatter.I18n.getText("jira.wiki.editor.operation.preformatted.placeholder")],
        'cite': ["??", "??", formatter.I18n.getText("jira.wiki.editor.operation.cite.placeholder")],
        'block-quote': ["{quote}", "{quote}", formatter.I18n.getText("jira.wiki.editor.operation.quote.block.placeholder")],
        'color': ["{color:$1}", "{color}", formatter.I18n.getText("jira.wiki.editor.operation.color.placeholder"), "red"], // 4th parameter is a default value to replace $
        'link': ["[", "|http://example.com]", formatter.I18n.getText("jira.wiki.editor.operation.link.external.placeholder")],
        'link-anchor': ["[#", "]", formatter.I18n.getText("jira.wiki.editor.operation.link.anchor.placeholder")],
        'link-mail': ["[mailto:", "]", "mail@example.com"],
        'image': ["!", "!", "http://example.com/image.png"],
        'mention': ["@", "", formatter.I18n.getText("jira.wiki.editor.operation.userMention.placeholder")],
        'code': ["\n{code:java}\n","\n{code}\n", "// Some comments here\npublic String getFoo()\n{\n    return foo;\n}"],
        'noformat': ["\n{noformat}\n", "\n{noformat}\n", formatter.I18n.getText("jira.wiki.editor.operation.noFormat.placeholder")],
        'panel': ["\n{panel:title="+formatter.I18n.getText("jira.wiki.editor.operation.panel.placeholder.title")+"}\n", "\n{panel}\n", formatter.I18n.getText("jira.wiki.editor.operation.panel.placeholder")],
        'table': ["\n||"+formatter.I18n.getText("jira.wiki.editor.operation.table.placeholder.heading")+" 1||"+formatter.I18n.getText("jira.wiki.editor.operation.table.placeholder.heading")+" 2||\n|", "|"+formatter.I18n.getText("jira.wiki.editor.operation.table.placeholder.column")+" A2|\n", formatter.I18n.getText("jira.wiki.editor.operation.table.placeholder.column")+" A1"]
    };
    wrapperMarkup['monospace-inline'] = wrapperMarkup['monospace'];

    for(var key in wrapperMarkup) {
        WikiEnabledTextarea.prototype[key] = (function(args, operationKey) {
            var paramsNumber = 0;
            if (args.length > 3) {
                paramsNumber = args.length - 3;
            }
            return function() {

                var prefix = args[0];
                var suffix = args[1];
                var placeholder = args[2];

                var i;
                for (i = 1; i <= paramsNumber; i++) {
                    prefix = prefix.replace("$"+i, arguments[i-1] || args[2 + i]);
                    suffix = suffix.replace("$"+i, arguments[i-1] || args[2 + i]);
                }

                var value = this.el.value;
                var selection = this.manipulationEngine.getSelection();

                this.undoRedoEl.recordHistoryItem();

                // unwrap selection if needed
                if (value.substring(selection.start - prefix.length, selection.start) === prefix &&
                    value.substring(selection.end, selection.end + suffix.length) === suffix) {

                    this.el.value = value.substring(0,selection.start - prefix.length) +
                    value.substring(selection.start, selection.end) +
                    value.substring(selection.end + suffix.length, value.length);

                    this.manipulationEngine.setSelection(selection.start - prefix.length, selection.end - prefix.length);

                    // wrap selection with markup
                } else {
                    this.manipulationEngine.wrapSelectionWith.call(this.manipulationEngine, prefix, suffix, placeholder);
                }
                this.undoRedoEl.recordHistoryItem();
                this.$el.trigger("input");
            };
        })(wrapperMarkup[key], key);
    }

    /**
     * Markup that is always a prefix for entire line, like h1. or bullet lists "* ".
     * Those prefixes are mutually exclusive, which means each line can be only prefixed with one of them
     * and applying one of them will remove any other.
     *
     * Each of this properties gets a function on the prototype with the same name
     * that changes value of textarea to apply the markup.
     */
    var excludingLinePrefixMarkup = {
        'paragraph': "",
        'bullet-list': "* ",
        'numbered-list': "# ",
        'h1': 'h1. ',
        'h2': 'h2. ',
        'h3': 'h3. ',
        'h4': 'h4. ',
        'h5': 'h5. ',
        'h6': 'h6. ',
        'paragraph-quote': 'bq. '
    };

    for(var key in excludingLinePrefixMarkup) {
        WikiEnabledTextarea.prototype[key] = (function(prefix) {
            return function() {
                this.undoRedoEl.recordHistoryItem();
                // check if line is already prefixed with one of the prefixes there
                for(var prefixToRemove in excludingLinePrefixMarkup) {
                    if (this.manipulationEngine.areSelectedLinesPrefixed(excludingLinePrefixMarkup[prefixToRemove])) {
                        this.manipulationEngine.unprefixSelectedLines(excludingLinePrefixMarkup[prefixToRemove]);
                    }
                }

                this.manipulationEngine.prefixSelectedLines.call(this.manipulationEngine, prefix);
                this.undoRedoEl.recordHistoryItem();
                this.$el.trigger("input");
            };
        })(excludingLinePrefixMarkup[key]);
    }

    /**
     * Markup that is just inserted in place of cursor or selection. The cursor is then placed at the end of inserted text.
     *
     * Each of this properties gets a function on the prototype with the same name
     * that changes value of textarea to apply the markup.
     */
    var insertMarkupWithoutSelecting = {
        'hr': "\n----\n",
        ':)': ':)',
        ':(': ':(',
        ':P': ':P',
        ':D': ':D',
        ';)': ';)',
        '(y)': '(y)',
        '(n)': '(n)',
        '(i)': '(i)',
        '(/)': '(/)',
        '(x)': '(x)',
        '(!)': '(!)',
        '(+)': '(+)',
        '(-)': '(-)',
        '(?)': '(?)',
        '(on)': '(on)',
        '(off)': '(off)',
        '(*)': '(*)',
        '(*r)': '(*r)',
        '(*g)': '(*g)',
        '(*b)': '(*b)',
        '(*y)': '(*y)',
        '(flag)': '(flag)',
        '(flagoff)': '(flagoff)',
        'image-attachment': [" !$1|thumbnail! ", "attachment-name.jpg"],
        'image-attachment-full': [" !$1! ", "attachment-name.gif"],
        'link-attachment': [" [^$1] ", "attachment-name.zip"]
    };

    for(var key in insertMarkupWithoutSelecting) {
        WikiEnabledTextarea.prototype[key] = (function(toInsert) {
            var paramsNumber = 0;
            var defaultValues;
            if (_.isArray(toInsert)) {
                paramsNumber = toInsert.length - 1;
                defaultValues = toInsert.slice(1);
                toInsert = toInsert[0];
            }
            return function() {
                var i;
                var result = toInsert;
                for (i = 1; i <= paramsNumber; i++) {
                    result = toInsert.replace("$"+i, arguments[i-1] || defaultValues[i-1]);
                }
                this.undoRedoEl.recordHistoryItem();
                this.manipulationEngine.replaceSelectionWith.call(this.manipulationEngine, result, false);
                this.undoRedoEl.recordHistoryItem();
                this.$el.trigger("input");
            };
        })(insertMarkupWithoutSelecting[key]);
    }

    WikiEnabledTextarea.prototype.speech = function()
    {
        speechRecognition.start(this);
    };

    return WikiEnabledTextarea;
});