define('wiki-edit/WikiEditorUtil', [
    'wiki-edit/viewissue/templates',
    'jira/util/formatter',
    'jquery',
    'underscore'
], function(ViewIssueTemplates, formatter, $, _) {

    var _defaultAttachmentTypeIcon = {
        cssClass : 'aui-iconfont-devtools-file',
        title : formatter.I18n.getText('jira.wiki.editor.operation.attachment.icon.file')
    };

    var _fileExtToAttachmentTypeIconCssClass = {
        'pdf'   : 'aui-iconfont-file-pdf',

        'gif'   : 'aui-iconfont-image',
        'png'   : 'aui-iconfont-image',
        'jpeg'  : 'aui-iconfont-image',
        'jpg'   : 'aui-iconfont-image',

        'xml'   : 'aui-iconfont-file-code',
        'html'  : 'aui-iconfont-file-code',
        'htm'   : 'aui-iconfont-file-code',
        'txt'   : 'aui-iconfont-file-txt',

        'zip'   : 'aui-iconfont-file-zip',
        'gz'    : 'aui-iconfont-file-zip',
        'tar'   : 'aui-iconfont-file-zip',
        'rar'   : 'aui-iconfont-file-zip',
        '7z'    : 'aui-iconfont-file-zip',

        'doc'   : 'aui-iconfont-file-doc',
        'docx'  : 'aui-iconfont-file-doc',
        'xls'   : 'aui-iconfont-file-xls',
        'xlsx'  : 'aui-iconfont-file-xls',
        'xlsm'  : 'aui-iconfont-file-xls',
        'ppt'   : 'aui-iconfont-file-ppt',
        'pptx'  : 'aui-iconfont-file-ppt',

        'java'  : 'aui-iconfont-file-code',
        'c'     : 'aui-iconfont-file-code',
        'h'     : 'aui-iconfont-file-code',
        'cpp'   : 'aui-iconfont-file-code',
        'hpp'   : 'aui-iconfont-file-code',
        'scala' : 'aui-iconfont-file-code',
        'php'   : 'aui-iconfont-file-code',
        'css'   : 'aui-iconfont-file-code',
        'less'  : 'aui-iconfont-file-code',
        'soy'   : 'aui-iconfont-file-code',
        'js'    : 'aui-iconfont-file-code',
        'jar'   : 'aui-iconfont-file-zip',
        'war'   : 'aui-iconfont-file-zip',
        'obr'   : 'aui-iconfont-file-zip'
    };

    var _fileExtToAttachmentTypeIconTitle = {
        'pdf'   : formatter.I18n.getText('icons.pdf.file'),

        'gif'   : formatter.I18n.getText('icons.gif.file'),
        'png'   : formatter.I18n.getText('icons.png.file'),
        'jpeg'  : formatter.I18n.getText('icons.jpeg.file'),
        'jpg'   : formatter.I18n.getText('icons.jpeg.file'),

        'xml'   : formatter.I18n.getText('icons.xml.file'),
        'html'  : formatter.I18n.getText('icons.html.file'),
        'htm'   : formatter.I18n.getText('icons.html.file'),
        'txt'   : formatter.I18n.getText('icons.text.file'),

        'zip'   : formatter.I18n.getText('icons.zip.archive'),
        'gz'    : formatter.I18n.getText('icons.gzip.archive'),

        'doc'   : formatter.I18n.getText('icons.microsoft.word'),
        'docx'  : formatter.I18n.getText('icons.microsoft.word'),
        'xls'   : formatter.I18n.getText('icons.microsoft.excel'),
        'xlsx'  : formatter.I18n.getText('icons.microsoft.excel'),
        'xlsm'  : formatter.I18n.getText('icons.microsoft.excel'),
        'ppt'   : formatter.I18n.getText('icons.microsoft.powerpoint'),
        'pptx'  : formatter.I18n.getText('icons.microsoft.powerpoint'),

        'java'  : formatter.I18n.getText('icons.java.source.file'),
        'jar'   : formatter.I18n.getText('icons.java.archive.file'),
        'war'   : formatter.I18n.getText('icons.java.archive.file')
    };

    return {

        /**
         * Default attachment type icon, used as a fallback if type couldn't be determined
         */
        defaultAttachmentTypeIcon : _defaultAttachmentTypeIcon,

        canTranslateMimeTypeToAttachmentTypeIcon: function() {
            return !!ViewIssueTemplates;
        },

        /**
         * Used by attachments dropdown to determine what icon to put to the left of attachment name
         * Icon is decided by attachment mime type
         *
         * @param {string} mimeType attachment mime type (eg: image/jpg)
         *
         * @returns Object with cssClass and title attributes.
         *  If translation fails cssClass is aui-iconfont-devtools-file and title is empty string.
         */
        translateMimeTypeToAttachmentTypeIcon : function (mimeType) {
            return {
              cssClass : ViewIssueTemplates.matchFileClass( { mimetype : mimeType } ),
              title : ViewIssueTemplates.matchFileIconAlt( { mimetype : mimeType } )
            };
        },

        /**
         * Used by attachments dropdown to determine what icon to put to the left of attachment name
         * Icon is decided by file extension
         *
         * @param extension extension name, with or without dot prefix (eg: .pdf, doc)
         *
         * @returns Object with cssClass and title attributes.
         *  If translation fails cssClass is aui-iconfont-devtools-file and title is empty string.
         */
        translateFileExtensionToAttachmentTypeIcon : function (extension) {
            var result = _.clone(_defaultAttachmentTypeIcon);

            if (typeof extension !== 'string') {
                return result;
            }

            extension = extension.toLowerCase();
            extension = extension.replace(/^\.+/, '');

            var cssClass = _fileExtToAttachmentTypeIconCssClass[extension];
            if (typeof cssClass === 'string') {
                result.cssClass = cssClass;
            }

            var title = _fileExtToAttachmentTypeIconTitle[extension];
            if (typeof title === 'string') {
                result.title = title;
            }

            return result;
        },

        /**
         * Perform version number comparison. Returns a number less than 0 if version A is lower than B, greater than 0
         * if A is higher than B, 0 if both equals.
         *
         * Example:
         * - compareVersion('5.7.10', '5.8.0') // returns -1
         * - compareVersion('5.8.0', '5.8.0') // returns 0
         * - compareVersion('5.8.1', '5.8.0') // returns 1
         * - compareVersion('5.10.0', '5.8.0') // returns 1
         * - compareVersion('5.8', '5.8.0') // returns -1
         * - compareVersion('5.8', '5.8.0') // returns -1
         * - compareVersion('5.0a', '5.0') // returns NaN
         *
         * @param verA (string) version number (eg: 5.7.10)
         * @param verB (string) version number (eg: 5.7.10)
         *
         * @return (number) A number less than 0 if version A is lower than B, greater than 0 if A is higher than B,
         * 0 if both equals, NaN if version comparison can't be performed
         */
        compareVersion : function (verA, verB) {
            var partsA = verA.split('.');
            var partsB = verB.split('.');
            var result = NaN;

            $(partsA).each(function (i, e) {
                if (i >= partsB.length) {
                    return false;
                }
                var partA = Number(e);
                var partB = Number(partsB[i]);
                if (isNaN(partA) || isNaN(partB)) {
                    //noinspection JSUnusedAssignment
                    result = NaN;
                    return false;
                }
                if (partA < partB) {
                    //noinspection JSUnusedAssignment
                    result = -1;
                    return false;
                } else if (partA > partB) {
                    //noinspection JSUnusedAssignment
                    result = 1;
                    return false;
                }
                result = 0;
            });

            if (result == 0 && partsA.length != partsB.length) {
                return (partsA.length > partsB.length) ? 1 : -1;
            }

            return result;
        },

        /**
         * Returns the file extension (without dot prefix) of a given file name, or an empty string if unable to do so
         * @param fileName
         */
        getFileExtension : function (fileName) {
            if (typeof fileName !== "string") {
                return "";
            }

            var matches = fileName.match(/\.(\w{1,5})$/i); // regex match exts such as .gz,.jpg,.docx,.3gp and discard the dot prefix
            if (matches && matches.length > 1) {
                return matches[1];
            }

            return "";
        }
    }
});
