define('wiki-edit/WikiEditor', [
    'wiki-edit/WikiEnabledTextarea',
    'jquery',
    'underscore',
    'wiki-edit/keymaster',
    'wiki-edit/analytics',
    'wiki-edit/WikiEditorUtil',
    'wiki-edit/editor/templates',
    'jira/util/navigator',
    'jira/util/events',
    'jira/util/events/types',
    'jira/editor/constants',
    'wrm/data',
    'wiki-edit/thumbnails',
    'wiki-edit/attachment-list-util'
], function (WikiEnabledTextarea,
             $,
             _,
             key,
             analytics,
             WikiEditorUtil,
             WikiEditorTemplates,
             Navigator,
             JIRAEvents,
             Types,
             EditorConstants,
             wrmData,
             Thumbnails,
             AttachmentListUtil) {
    var helpData = wrmData.claim('com.atlassian.jira.plugins.jira-wiki-editor:wiki-editor-resources.help-data');

    var controlKey = 'Ctrl';
    if (Navigator.isMac()) {
        controlKey = '⌘';
    }

    var determineFileType = function (fileName) {
        return fileName.match(/\.(bmp|gif|jpg|jpeg|pjpeg|png|tiff|webp)$/i) ? 'image' : 'file';
    };

    var insertMarkup = function (wikiEditor, fileName, fileType, thumbnailable, fileHref) {
        if (wikiEditor.$el.triggerHandler(EditorConstants.EVENT_OPERATION_ATTACHMENT, {
                name: fileName,
                type: fileType,
                thumbnailable: thumbnailable,
                href: fileHref
            })) {
            return;
        }

        if (fileType == "image") {
            if (thumbnailable) {
                wikiEditor["image-attachment"](fileName);
            } else {
                wikiEditor["image-attachment-full"](fileName);
            }

            analytics.trigger("operation.attachment.image");
        } else {
            wikiEditor["link-attachment"](fileName);
            analytics.trigger("operation.attachment.link");
        }
    };

    function _closeDropDown(eventTarget) {
        var $theContainer = $(eventTarget).closest(".aui-dropdown2");
        if ($theContainer.length) {
            $theContainer[0].hide();
        }
    }

    return {
        _isThumbnailable: Thumbnails.isThumbnailable,

        create: function (element) {
            // disable plugin in IE <=8
            if (typeof element.selectionStart == "undefined") {
                return;
            }

            var $editorElement = $(element);

            if ($editorElement.data("wikiEditor")) {
                return;
            }

            analytics.trigger("initialized");

            // todo editor should initialize all html and events by itself
            var wikiEditor = new WikiEnabledTextarea(element);
            this.wikiEditor = wikiEditor;

            var $editorHtml = $(WikiEditorTemplates.renderEditor({
                editorId: wikiEditor.id,
                controlKey: controlKey,
                helpData: helpData
            }));
            var $editorToolbar = $editorHtml.filter(".wiki-edit-toolbar");

            // speech-capable browser?
            if ("webkitSpeechRecognition" in window) {
                $editorHtml.find(".wiki-edit-speech-item").removeClass("hidden");
            }

            $editorElement.parent().before($editorHtml);

            // detect if we are dealing with dialog
            var isDialog = $editorElement.closest(".jira-dialog").length > 0 || $editorElement.closest("form#issue-create").length > 0;
            var isCreateDialog = $editorElement.closest("#create-issue-dialog").length > 0;

            var isViewIssue = $("#issue-content").length > 0;
            var isComment = $editorElement.closest("#addcomment").length > 0;

            var elementsWithTooltips = $editorHtml.find(".wiki-edit-operation[title], .aui-dropdown2-trigger, .wiki-edit-tooltip");
            var onWikiEditorHideEventName;
            if (isDialog) {
                onWikiEditorHideEventName = 'Dialog.beforeHide';
            } else {
                onWikiEditorHideEventName = Types.BEFORE_INLINE_EDIT_CANCEL;
            }
            elementsWithTooltips.each(function () {
                var $this = $(this);
                var config = {
                    aria: true,
                    hideOnClick: true,
                    hoverable: false
                };
                if ($this.parent().is('li')) {
                    config.gravity = 'w';
                }
                $this.tooltip(config);
            });
            var hideAllTooltips = function () {
                elementsWithTooltips.each(function () {
                    $(this).tooltip('hide');
                });
            };
            JIRAEvents.bind(onWikiEditorHideEventName, hideAllTooltips);

            var isDropdownEnabled = false;
            var $wikiDropdown = $editorHtml.filter(".aui-dropdown2");
            $wikiDropdown.on({
                'aui-dropdown2-show': function () {
                    isDropdownEnabled = true;

                    // focus on editorElement when dropdown opens so
                    // it can handle keyboard event (especially in dialogs)
                    if (!$editorElement.hasClass(EditorConstants.CLASS_COVER)) {
                        $editorElement.focus();
                    }
                },
                'aui-dropdown2-hide': function () {
                    isDropdownEnabled = false;

                    hideAllTooltips();
                }
            });
            $editorElement.on('keydown', function checkEscapePressWhenDropdownIsEnabled(e) {
                if (isDropdownEnabled && e.keyCode === AJS.keyCode.ESCAPE) {
                    if (isDialog) {
                        JIRAEvents.one("Dialog.beforeHide", function (e) {
                            e.preventDefault();
                        });
                    } else if (isViewIssue && !isComment) {
                        JIRAEvents.one(Types.BEFORE_INLINE_EDIT_CANCEL, function preventInlineEditCloseIfAnyDropdownIsEnabled(e) {
                            e.preventDefault();
                        });
                    }
                }
            });

            var $attachmentDropdown = $editorHtml.filter(
                ".wiki-edit-dropdown-image, " +
                ".wiki-edit-dropdown-attachment"
            );

            var $attachmentListContainer = $attachmentDropdown.find(
                ".wiki-edit-insert-image, " +
                ".wiki-edit-insert-attachment"
            );

            var $attachmentList = $attachmentDropdown.find(
                ".wiki-edit-insert-image ul, " +
                ".wiki-edit-insert-attachment ul"
            );

            // Attachment dropdown link click handler
            $attachmentList.on("click", "a", function (e) {
                e.preventDefault();
                _closeDropDown(e.target);
                var fileType = $(this).attr("data-attachment-type");
                if (typeof fileType === "undefined") {
                    fileType = "file";
                }

                var fileName = $(this).find('.wiki-edit-attachment-name').text();

                var thumbnailable = Thumbnails.isThumbnailable(fileName, $(this));

                insertMarkup(wikiEditor, fileName, fileType, thumbnailable, $(this).attr("data-attachment-href"));
            });

            // Build attachment links when attachment button is pressed
            $attachmentDropdown.on("aui-dropdown2-show", function () {
                AttachmentListUtil.buildAttachmentLinks($attachmentList, $attachmentListContainer, isDialog, isCreateDialog, isViewIssue);
            });

            // dropdown2 fix so when item is selected using keyboard the container div scrolls
            // properly. This bug was fixed on AUI 5.8+ so only do this for older AUI versions
            if (WikiEditorUtil.compareVersion(AJS.version, '5.8.0') < 0) {
                $attachmentDropdown.on('aui-dropdown2-item-selected', function scrollContainerDivIfActiveItemIsOutOfView() {
                    var $active = $(this).find('.aui-dropdown2-active');
                    if (typeof $active !== 'object') {
                        return;
                    }
                    var $container = $active.closest('div');
                    if (typeof $container !== 'object') {
                        return;
                    }

                    // top/bottom limits are extreme points above/below which element will not / only be partially visible
                    var containerTopLimit = $container.offset().top;
                    var containerBottomLimit = containerTopLimit + $container.height();

                    var elementTop = $active.offset().top;
                    var elementBottom = elementTop + $active.height();

                    // Active element is (partially) above the container viewport, scroll up
                    if (elementTop < containerTopLimit) {
                        $container.scrollTop($container.scrollTop() - (containerTopLimit - elementTop));
                    }

                    // Active element is (partially) below the container viewport, scroll down
                    if (elementBottom > containerBottomLimit) {
                        $container.scrollTop($container.scrollTop() + (elementBottom - containerBottomLimit));
                    }
                });
            }

            $editorHtml.filter(".wiki-edit-dropdown, .wiki-edit-picker").on("mousedown", function (e) {
                e.preventDefault(); // prevent blurring textarea, needed for inline mode
            });

            // attach keyboard shortcuts
            $editorHtml.find(".wiki-edit-operation:not(.wiki-edit-operation-color)").each(function () {
                var $this = $(this);
                var handler = function (e) {
                    if (!e) {
                        return;
                    }
                    e.preventDefault && e.preventDefault();
                    _closeDropDown(e.target);

                    if (wikiEditor.$el.triggerHandler(EditorConstants.EVENT_OPERATION, $this.data("operation"))) {
                        return;
                    }

                    $editorElement.focus();
                    wikiEditor[$this.data("operation")]();
                    analytics.trigger("operation", {
                        name: $this.data("operation"),
                        nameHash: analytics.hashCode($this.data("operation")),
                        trigger: e.type
                    });
                    analytics.trigger("operation." + $this.data("operation"), {trigger: e.type});
                };

                // todo event delegation, fool!
                $this.click(handler);
                if ($this.data("shortcut")) {
                    // todo shortcuts should be binded once, globally
                    key($this.data("shortcut"), wikiEditor.id, handler);
                }
            });

            // text color
            $editorHtml.find(".wiki-edit-operation-color").each(function () {
                var $this = $(this);
                var handler = function (e) {
                    if (!e) {
                        return;
                    }
                    e.preventDefault && e.preventDefault();
                    _closeDropDown(e.target);

                    if (wikiEditor.$el.triggerHandler(EditorConstants.EVENT_OPERATION_COLOR, $this.data("color"))) {
                        return;
                    }

                    $editorElement.focus();
                    wikiEditor[$this.data("operation")]($this.data("color"));
                    analytics.trigger("operation", {
                        name: $this.data("operation"),
                        nameHash: analytics.hashCode($this.data("operation")),
                        trigger: e.type
                    });
                    analytics.trigger("operation." + $this.data("operation"), {trigger: e.type});
                };

                // todo event delegation, fool!
                $this.click(handler);
                if ($this.data("shortcut")) {
                    // todo shortcuts should be binded once, globally
                    key($this.data("shortcut"), wikiEditor.id, handler);
                }

            });

            var defaultColor = (window.localStorage.getItem("jira.wikieditor.last.color")) ? window.localStorage.getItem("jira.wikieditor.last.color") : "red";
            var $colorOperation = $editorHtml.find(".wiki-edit-operation-color");
            $colorOperation.data("color", defaultColor);
            var $colorIndicator = $editorToolbar.find(".wiki-edit-color-indicator");
            $colorIndicator.css("background-color", defaultColor);

            $editorHtml.filter(".wiki-edit-color-picker").on("click", "a", function (e) {
                e.preventDefault();
                _closeDropDown(e.target);
                var color = $(this).data("color");
                $colorOperation.data("color", color);
                $colorIndicator.css("background-color", color);
                window.localStorage.setItem("jira.wikieditor.last.color", color);

                if (wikiEditor.$el.triggerHandler(EditorConstants.EVENT_OPERATION_COLOR, color)) {
                    return;
                }

                $editorElement.focus();
                wikiEditor.color(color);
                analytics.trigger("operation", {name: "color"});
                analytics.trigger("operation.color.picked", {color: color});
            });

            var $visibilityToggle = $editorHtml.find('.wiki-edit-visible-toggle');
            var hideToolbar = (localStorage.getItem("jira.wikieditor.visibility" + element.name + '.' + element.id) === "true") ? true : false;
            if (hideToolbar) {
                $visibilityToggle.addClass("active");
                $editorToolbar.addClass("wiki-edit-toolbar-compact");
            }

            function toggleEditor(e) {
                e.preventDefault();
                e.stopPropagation();
                hideToolbar = !hideToolbar;
                $visibilityToggle.toggleClass("active", hideToolbar);
                $editorToolbar.toggleClass("wiki-edit-toolbar-compact", hideToolbar);
                localStorage.setItem("jira.wikieditor.visibility" + element.name + '.' + element.id, (hideToolbar) ? "true" : "false");
                analytics.trigger((hideToolbar) ? "hide" : "show")
            }

            $visibilityToggle.on("click", toggleEditor);
            $editorToolbar.parent().on("click", ".wiki-edit-toolbar-compact", toggleEditor);

            $(document).on("showWikiPreview", function (e, el) {
                if ($(el).find(element).length) {
                    $editorToolbar.addClass("hidden");
                }
            });
            $(document).on("showWikiInput", function (e, el) {
                if ($(el).find(element).length) {
                    $editorToolbar.removeClass("hidden");
                }
            });
        },

        initBrowseLink: function (element) {
            var wikiEditor = this.wikiEditor;

            var $editorElement = $(element);

            var $browseButton = $editorElement.parent().siblings().find(".wiki-attachment-browse");
            var canAttach = $("#dnd-metadata-webpanel").data("can-attach");

            var isDialog = $editorElement.closest(".jira-dialog").length > 0 || $editorElement.closest("form#issue-create").length > 0;

            if (isDialog) {
                // These are the dialogs we want to show the browse link in.
                // The dialogs have an attachment dropzone that we can use to simulate a drop.
                var supportedDialogSelectors = ['#create-issue-dialog:visible', '#create-subtask-dialog:visible', '#edit-issue-dialog:visible', '#create-linked-issue-dialog:visible'];
                var supportedDialog = $(supportedDialogSelectors.join(","));
                var isSupportedDialog = supportedDialog.length > 0;
                canAttach = $(".jira-dialog fieldset span:contains('Attachment')").length > 0;

                // If we did not find a dialog that we know can receive files, we search
                // for other dialogs that explicitly state that they can receive files.
                if (!isSupportedDialog) {
                    supportedDialog = $editorElement.closest(".dialog-can-receive-files:visible");
                    isSupportedDialog = supportedDialog.length > 0;
                    // If we do find such a dialog, we consume its can-attach (boolean) data attribute.
                    // If the dialog does not have that attribute, canAttach will be undefined and the
                    // browse link will not be shown.
                    canAttach = supportedDialog.data("can-attach");
                }
                var insertMarkupInDialog = supportedDialog.data("insert-markup-on-browse");
            }

            var $attachmentsDropdown = $editorElement.parent().siblings().find(".wiki-edit-attachment-picker-trigger");

            if (canAttach && (isDialog ? isSupportedDialog : true)) {
                $browseButton.closest("div").removeClass("hidden");
                $attachmentsDropdown.removeClass("hidden");
                if (!$browseButton.data("initialised")) {
                    $browseButton.on('click', function (e) {
                        e.preventDefault();
                        _closeDropDown(e.target);

                        var $tempFileInput = $("<input type=\"file\" multiple />");
                        $tempFileInput.change(function (event) {
                            var files = $tempFileInput[0].files;
                            if (isDialog) {
                                var dropEvent = $.Event("drop");
                                dropEvent.dataTransfer = {files: files};
                                supportedDialog.find('.issue-drop-zone__target').trigger(dropEvent);
                            }
                            else {
                                // This event comes from the JIRA Drag and Drop Plugin.
                                JIRAEvents.trigger("attachmentForPageReceived", {
                                    files: files,
                                    isWikiTextfieldFocused: false,
                                    wikiTextfield: element
                                });
                            }

                            if (wikiEditor.$el.triggerHandler(EditorConstants.EVENT_OPERATION_ATTACHMENT, _.map(files, function (file) {
                                    return {
                                        name: file.name,
                                        type: determineFileType(file.name),
                                        thumbnailable: Thumbnails.isThumbnailable(file.name),
                                        href: window.URL.createObjectURL(file)
                                    };
                                }))) {
                                return;
                            }

                            if (isDialog ? insertMarkupInDialog : true) {
                                for (var i = 0; i < files.length; i++) {
                                    var fileName = files[i].name;
                                    insertMarkup(wikiEditor, fileName, determineFileType(fileName), Thumbnails.isThumbnailable(fileName), window.URL.createObjectURL(files[i]));
                                }
                            }
                        });
                        $tempFileInput.click();

                        analytics.trigger("attachment.dropdown.button.clicked.browse");
                    });
                    $browseButton.data("initialised", true);
                }
            }
            else {
                $browseButton.closest("div").addClass("hidden");

                // Here we know the browse button will be hidden. If we will
                // not show an attachments list, the attachment dropdown will be empty
                // so we want to hide the attachment dropdown altogether.
                var $genericAttachments = $("#attachment_thumbnails li.attachment-content, #file_attachments li.attachment-content");
                var currentIssueHasAttachments = $genericAttachments.length > 0;

                // Create issue/Create subtask dialogs are special because even if the current issue has attachments,
                // we are not supposed to show these attachments in those two dialogs (because attachments are per issue).
                var isCreateIssueOrSubtaskDialog = isDialog && isSupportedDialog &&
                    (supportedDialog.attr("id") === "create-issue-dialog" || supportedDialog.attr("id") === "create-subtask-dialog");

                var willShowAttachmentsList = currentIssueHasAttachments && !isCreateIssueOrSubtaskDialog;

                if (!willShowAttachmentsList) {
                    $attachmentsDropdown.addClass("hidden");
                }
            }

        }
    }
});

