define('wiki-edit/UndoManager', [
    'backbone',
    'underscore'
], function(
    Backbone,
    _
) {

    /**
     * Simple undo/redo manager.
     * Basically the structure is like [---undoList---][CURRENT][---redoList---].
     * All operations move the current element on that "concatenated" list.
     */
    var UndoManager =  function() {
        this._undoStack = [];
        this._redoStack = [];
        this._current = undefined;

        _.extend(this, Backbone.Events);
    };

    UndoManager.EventTypes = UndoManager.prototype.EventTypes = {
        UNDO: "UndoManager:undo",
        REDO: "UndoManager:redo",
        CLEAR: "UndoManager:clear",
        PUSH: "UndoManager:push",
        UPDATE_CURRENT: "UndoManager:updateCurrent"
    };

    UndoManager.prototype.MAX_STACK = 1000;
    /**
     * Sets the current state of the manager
     * @param {Anything} current - state
     * @param {Boolean} preserveRedo - if set to true existing redo entries will be preserved
     */
    UndoManager.prototype.updateCurrent = function(current, preserveRedo) {
        if (current != this._current) {
            this._current = current;
            if(!preserveRedo) {
                this._redoStack.length = 0; // reset the redo stack
            }

            this.trigger(this.EventTypes.UPDATE_CURRENT);
        }
    };

    /**
     * Add val to the undo stack and make it undoable
     * @param {Anything} val - state to be made undoable
     */
    UndoManager.prototype.push = function(val) {
        this._redoStack.length = 0; // reset the redo stack
        if (!this._undoStack.length || !_.isEqual(val, this._undoStack[this._undoStack.length - 1])) {
            this._undoStack.push(val);
            this._current = val;
        }
        if (this._undoStack.length > this.MAX_STACK) {
            this._undoStack.splice(0, this._undoStack.length - this.MAX_STACK)
        }

        this.trigger(this.EventTypes.PUSH);
    };

    /**
     * Undo a value. Set it to be the current element and push the current element to the redo stack
     * @return {Anything} current value after the undo operation
     */
    UndoManager.prototype.undo = function() {
        if (!this.canUndo()) { return; }
        var val;
        do {
            val = this._undoStack.pop();
        } while (_.isEqual(val, this._current) && this.canUndo());
        this._redoStack.push(this._current);
        this._current = val;

        this.trigger(this.EventTypes.UNDO);
        return val;
    };

    /**
     * Redo a value. Set it to be the current element and push the current element to the undo stack
     * @return {Anything} current value after the redo operation
     */
    UndoManager.prototype.redo = function() {
        if (!this.canRedo()) { return; }

        var val = this._redoStack.pop();
        this._undoStack.push(this._current);
        this._current = val;

        this.trigger(this.EventTypes.REDO);
        return val;
    };

    /**
     * Check if it is possible to undo
     * @return {Boolean}
     */
    UndoManager.prototype.canUndo = function() {
        return !!this._undoStack.length;
    };

    /**
     * Check if it is possible to redo
     * @return {Boolean}
     */
    UndoManager.prototype.canRedo = function() {
        return !!this._redoStack.length;
    };

    /**
     * Clears undo and redo stack
     */
    UndoManager.prototype.clear = function() {
        this._undoStack = [];
        this._redoStack = [];

        this.trigger(this.EventTypes.CLEAR);
    };

    return UndoManager;
});
