define('wiki-edit/TextareaManipulator', function() {

    var TextareaManipulator = function(element) {
        this.el = element;
    };

    TextareaManipulator.prototype.NEW_LINE_TOKEN = "\n";

    /**
     * Gets object containing selection info
     * @returns {{position: number, start: number, end: number, length: number, text: string}}
     */
    TextareaManipulator.prototype.getSelection = function() {
        return {
            start: this.el.selectionStart,
            end: this.el.selectionEnd,
            length: this.el.selectionEnd - this.el.selectionStart,
            text: this.el.value.substring(this.el.selectionStart, this.el.selectionEnd)
        };
    };

    /**
     * Sets selection
     * @param start {number} index in the string, start of the selection
     * @param end {number} index in the string, end of the selection
     */
    TextareaManipulator.prototype.setSelection = function(start, end) {
        if (arguments.length === 1) {
            end = start;
        }
        this.el.selectionStart = start;
        this.el.selectionEnd = end;
    };

    /**
     * Replaces selected text with a parameter, if no text is selected
     * it's inserted where the cursor is
     * @param text {string} text to replace selection with
     * @param selectReplaced {Boolean} indicates if inserted text should be selected
     */
    TextareaManipulator.prototype.replaceSelectionWith = function(text, selectReplaced) {
        var start = this.el.selectionStart;
        var val = this.el.value;
        this.el.value = val.substring(0, this.el.selectionStart) + text + val.substring(this.el.selectionEnd, val.length);

        this.el.selectionEnd = start + text.length;
        this.el.selectionStart = (selectReplaced) ? start : this.el.selectionEnd;
    };

    /**
     * Wraps selection with a prefix and a suffix but only if it isn't already wrapped
     * @param prefix {string}
     * @param suffix {string}
     * @param placeholder {string} text to put between prefix and suffix if no selection was made
     */
    TextareaManipulator.prototype.wrapSelectionWith = function(prefix, suffix, placeholder) {
        if (arguments.length < 3) {
            placeholder = "";
        }
        if (arguments.length === 1) {
            suffix = prefix;
        }
        var selection = this.getSelection();
        var val = this.el.value;

        if (val.substring(selection.start - prefix.length, selection.start) === prefix &&
            val.substring(selection.end, selection.end + suffix.length) === suffix) {
            return;
        }

        var middle = val.substring(selection.start, selection.end);
        if (middle.length == 0) {
            middle = placeholder;
        }
        this.el.value = val.substring(0, selection.start) + prefix + middle + suffix + val.substring(selection.end, val.length);
        var newSelectionStart = selection.start + prefix.length;
        this.setSelection(newSelectionStart, newSelectionStart + middle.length);
    };

    /**
     * Finds selected lines within the textarea and returns an object containing full contents of the object
     * as an array of lines and start and end indexes of the selected lines
     * @returns {{lines: Array, start: number, end: number}}
     * @private
     */
    TextareaManipulator.prototype._getSelectedLines = function() {
        var val = this.el.value;
        var selection = this.getSelection();
        var startLine = val.substring(0, selection.start).split(this.NEW_LINE_TOKEN).length - 1; // 0-based index
        var endLine = startLine + val.substring(selection.start, selection.end).split(this.NEW_LINE_TOKEN).length - 1; // 0-based index
        var lines = val.split(this.NEW_LINE_TOKEN);
        return {
            lines: lines,
            start: startLine,
            end: endLine
        }
    };

    /**
     * Returns line at which cursor is (or end of the selection)
     * @returns {string}
     */
    TextareaManipulator.prototype.getLineAtCursor = function() {
        var linesSelection = this._getSelectedLines();
        return linesSelection.lines[linesSelection.end];
    };

    /**
     * Return regexp match of the fist line in the selection
     * @param pattern {regexp} prefix to check against
     * @returns {string} matched string or {undefined}
     */
    TextareaManipulator.prototype.getFirstLineMatch = function(pattern) {
        var linesSelection = this._getSelectedLines();
        var match = linesSelection.lines[linesSelection.start].match(pattern);
        if (match) {
            return match[0];
        }
    };

    /**
     * Checks if any of the selected lines is prefixed with a given string
     * @param prefix {string} prefix to check against
     * @returns {boolean}
     */
    TextareaManipulator.prototype.areSelectedLinesPrefixed = function(prefix) {
        var linesSelection = this._getSelectedLines();
        var currentLine;

        for (currentLine = linesSelection.start; currentLine <= linesSelection.end; currentLine++) {
            if (linesSelection.lines[currentLine].indexOf(prefix) == 0) {
                return true;
            }
        }
        return false;
    };

    /**
     * Prefixes selected lines with a given parameter. Lines already containing the prefix remain unchanged
     * @param prefix {string}
     */
    TextareaManipulator.prototype.prefixSelectedLines = function(prefix) {
        var selection = this.getSelection();
        var linesSelection = this._getSelectedLines();
        var currentLine;
        var prefixedLines = 0;
        var firstLinePrefixed = true;
        for (currentLine = linesSelection.start; currentLine <= linesSelection.end; currentLine++) {
            if (linesSelection.lines[currentLine].indexOf(prefix) == 0) {
                if (currentLine == linesSelection.start) {
                    firstLinePrefixed = false;
                }
                continue; //don't prefix lines which already have ie
            }
            prefixedLines += 1;
            linesSelection.lines[currentLine] = prefix + linesSelection.lines[currentLine];
        }
        this.el.value = linesSelection.lines.join(this.NEW_LINE_TOKEN);

        var prefixesLength = prefix.length * prefixedLines;
        this.setSelection(selection.start + ((firstLinePrefixed) ? prefix.length : 0), selection.end + prefixesLength);
    };

    /**
     * Removes prefix from selected lines
     * @param prefix {string}
     */
    TextareaManipulator.prototype.unprefixSelectedLines = function(prefix) {
        var selection = this.getSelection();
        var linesSelection = this._getSelectedLines();
        var currentLine;
        var firstLinePrefix = false;
        var prefixedLines = 0;

        for (currentLine = linesSelection.start; currentLine <= linesSelection.end; currentLine++) {
            if (linesSelection.lines[currentLine].indexOf(prefix) == 0) {
                prefixedLines += 1;
                linesSelection.lines[currentLine] = linesSelection.lines[currentLine].substring(prefix.length);
                if (currentLine == linesSelection.start) {
                    firstLinePrefix = true;
                }
            }
        }
        this.el.value = linesSelection.lines.join(this.NEW_LINE_TOKEN);

        var prefixesLength = prefix.length * prefixedLines;
        this.setSelection(selection.start - ((firstLinePrefix) ? prefix.length : 0), selection.end - prefixesLength);
    };

    return TextareaManipulator;
});
