define('wiki-edit/attachment-list-util', [
    'wiki-edit/WikiEnabledTextarea',
    'jquery',
    'underscore',
    'wiki-edit/keymaster',
    'wiki-edit/analytics',
    'wiki-edit/WikiEditorUtil',
    'wiki-edit/editor/templates',
    'wiki-edit/thumbnails'
], function (WikiEnabledTextarea,
             $,
             _,
             key,
             analytics,
             WikiEditorUtil,
             WikiEditorTemplates,
             Thumbnails) {

    var determineFileType = function (fileName) {
        return fileName.match(/\.(bmp|gif|jpg|jpeg|pjpeg|png|tiff|webp)$/i) ? 'image' : 'file';
    };

    return {
        buildAttachmentLinks: function ($attachmentList, $attachmentListContainer, isDialog, isCreateDialog, isViewIssue) {
            // At the moment this will be filled with DOM elements of the attachments. Ideally this should be array of structured JSON object
            // so regardless what shape the DOM is (eg: SD and Agile renders attachments differently) the structure of the array is consistent
            var genericAttachments = [];

            // A list of attachments displayed on the issue. At the moment this is only used by Agile board and backlog view
            // Ideally this should be used everywhere editor can show up.
            //
            // The structure of elements dumped into this array is:
            // {
            //   name: "my-attachment-file-name.pdf",
            //   type: "file",             // or "image"
            //   icon: { cssClass : "aui-iconfont-file-pdf", title : "PDF File" }  // To easily create this use WikiEditorUtil.translateMimeTypeToAttachmentTypeIcon or WikiEditorUtil.translateFileExtensionToAttachmentTypeIcon
            // }
            //
            // NB: From JIRA 7.4.0 (atleast) it seems the way attachments are rendered in Agile views changed again, and we should use the genericAttachments approach for now! Trying handle both possibility at the moment
            var genericAttachmentsStructured = [];

            // Add temporary attachments (those added before dialog is saved such as in create/edit dialog)
            if (isDialog) {
                $(".upload-progress-bar:not(.upload-progress-bar__upload-error) span.upload-progress-bar__file-name").each(function (i, v) {
                    genericAttachments.push(v);
                });
            }

            var findAndAddGenericAttachments = function () {
                $("#attachment_thumbnails li.attachment-content, " +
                    "#file_attachments li.attachment-content").each(function (i, v) {
                    genericAttachments.push(v);
                });
            };

            // Add permanent attachments
            if (!isCreateDialog && isViewIssue) {
                findAndAddGenericAttachments();
            }

            // Add agile / greenhopper attachments -- for when editor pops up in agile board / backlog
            var buildAgileAttachmentObject = function (fileName, attachmentType) {
                if (!fileName) {
                    return null;
                }
                var fileExtension = WikiEditorUtil.getFileExtension(fileName);
                return {
                    name: fileName,
                    type: attachmentType,
                    icon: WikiEditorUtil.translateFileExtensionToAttachmentTypeIcon(fileExtension)
                };
            };

            var $agileAttachmentsContainer = $("#ghx-tab-attachments .ghx-container");
            var isAgileStructuredAttachmentView = $agileAttachmentsContainer.length > 0;
            if (isAgileStructuredAttachmentView) {
                $agileAttachmentsContainer.find("#file_attachments li.attachment-content > dl dt.attachment-title a, #file_attachments div.verbose dt.attachment-title a").each(function (i, e) {
                    var attachmentObj = buildAgileAttachmentObject(e.textContent, "file");
                    if (attachmentObj) {
                        genericAttachmentsStructured.push(attachmentObj);
                    }
                });
                $agileAttachmentsContainer.find("#attachment_thumbnails a.attachment-title").each(function (i, e) {
                    var attachmentObj = buildAgileAttachmentObject(e.textContent, "image");
                    if (attachmentObj) {
                        genericAttachmentsStructured.push(attachmentObj);
                    }
                });
            } else if ($("#ghx-detail-issue").length > 0) {
                // detect if we are on view issue page (in say issue details of sprint view)
                // but we only use this if we can't find the agile strucutred attachments (which seems to have changed)
                findAndAddGenericAttachments();
            }

            $attachmentList.empty();
            $attachmentListContainer.addClass("hidden");

            // This if block attempts to populate attachment list based on <li> element of each attachment displayed in issue view
            // This works for JIRA and SD view but not for Agile because Agile renders the attachment differently (but that seems to have changed, so it has generic attachments also)
            // Ideally the genericAttachment should be a list of structured JSON object instead (see genericAttachmentStructured array above)
            if ((isDialog || isViewIssue || !isAgileStructuredAttachmentView) && genericAttachments.length) {
                $attachmentListContainer.removeClass("hidden");

                /**
                 * Contents of $genericAttachments (ie: "this") could be:
                 * - A thumbnails (gallery) view item
                 * - A list view item
                 * - Create / edit dialog entry
                 */
                $(genericAttachments).each(function () {
                    // Handle various form of DOM structure a file name can appear in (create issue dialog,
                    // edit issue - list view, edit issue - thumbnail view
                    var $this = $(this);
                    var fileName = $this.is('span.upload-progress-bar__file-name') ?
                        $this.text() :
                        $this.find('a.attachment-title, .attachment-title a').first().text();

                    // Determine file type by 'data-attachment-type' attribute, or fall back by checking
                    // popular image file extension
                    var fileType = $this.attr('data-attachment-type');
                    if (fileType === undefined) {
                        fileType = determineFileType(fileName);
                    }

                    // Try to figure out the attachment type icon using mime type. On thumbnail / list view mime-type
                    // can be looked up using data-downloadurl attribute. If mime-type not available let's fall back
                    // using file extension. And if that still doesn't work use aui-iconfont-file-txt.
                    var typeIcon = WikiEditorUtil.defaultAttachmentTypeIcon;
                    var dataDownloadUrl = $this.attr('data-downloadurl');
                    if (dataDownloadUrl === undefined) {
                        dataDownloadUrl = $this.find('[data-downloadurl]').attr('data-downloadurl');
                    }
                    if (dataDownloadUrl !== undefined && typeof dataDownloadUrl === 'string' && WikiEditorUtil.canTranslateMimeTypeToAttachmentTypeIcon()) {
                        var attrs = dataDownloadUrl.split(':');
                        if (attrs.length > 0) {
                            var mimeType = attrs[0];
                            typeIcon = WikiEditorUtil.translateMimeTypeToAttachmentTypeIcon(mimeType);
                        }
                    } else if (fileName !== undefined && typeof fileName === 'string') {
                        var matches = fileName.match(/\.(\w{1,5})$/i); // regex match exts such as .gz,.jpg,.docx,.3gp and discard the dot prefix
                        if (matches && matches.length > 1) {
                            var fileExtension = matches[1];
                            typeIcon = WikiEditorUtil.translateFileExtensionToAttachmentTypeIcon(fileExtension);
                        }
                    }

                    var attachmentHref = $this.find('a.attachment-title, .attachment-title>a').attr("href");
                    //temporary attachment
                    if (!attachmentHref && $this.is('span.upload-progress-bar__file-name')) {
                        var temporaryAttachmentId = $this.parents('[data-file-id]').attr('data-file-id');
                        var formToken = $this.parents('form').find('input[name="formToken"]').attr('value') || 'undefined';

                        if (temporaryAttachmentId && formToken) {
                            attachmentHref = JIRA.Templates.Plugins.WikiEditor.temporaryAttachmentViewUrl({
                                formToken: formToken,
                                temporaryAttachmentId: temporaryAttachmentId,
                                filename: encodeURIComponent(fileName)
                            });
                        }
                    }

                    var thumbnailable = Thumbnails.isThumbnailable(fileName, $this);
                    $attachmentList.append(
                        $(WikiEditorTemplates.renderListItem({
                            text: fileName,
                            attachmentType: fileType,
                            attachmentTypeIcon: typeIcon,
                            attachmentThumbnailable: thumbnailable,
                            attachmentHref: attachmentHref
                        }))
                    );
                });

            } else if (!isCreateDialog && genericAttachmentsStructured.length > 0) {
                // A fallback if block if above is false. At the moment only used to ensure attachment lists are shown in Agile view.
                // Ideally the above if block should be removed
                $attachmentListContainer.removeClass("hidden");

                genericAttachmentsStructured.forEach(function (e) {
                    if (e == null || e.name == null) {
                        return;
                    }

                    var fileName = e.name;
                    var type = e.type ? e.type : "file";
                    var icon = e.icon ? e.icon : {cssClass: "aui-iconfont-devtools-file", title: ""};

                    var thumbnailable = _isThumbnailable(fileName);

                    $attachmentList.append(
                        $(WikiEditorTemplates.renderListItem({
                            text: fileName,
                            attachmentType: type,
                            attachmentThumbnailable: thumbnailable,
                            attachmentTypeIcon: icon
                        }))
                    );
                });
            }
        }
    };
});


