/*globals GH*/
define('jira/inline-issue-create/agile/initialiser', [
    'jira/util/formatter',
    'jira/util/logger',
    'jira/ajs/dark-features',
    'jira/jquery/deferred',
    'jquery',
    'jira/inline-issue-create/lib/underscore',
    'jira/inline-issue-create/util',
    'jira/inline-issue-create/lib/brace',
    'jira/inline-issue-create/entities/inline-bootstrap',
    'jira/inline-issue-create/agile/widget-group',
    'jira/inline-issue-create/agile/entities/issue-types',
    'jira/inline-issue-create/agile/views/description',
    'jira/inline-issue-create/views/default-trigger',
    'jira-agile/rapid/ui/plan/plan-controller',
    'jira-agile/rapid/ui/plan/backlog-controller',
    'jira-agile/rapid/ui/plan/backlog-model',
    'jira-agile/rapid/ui/plan/sprint-backlog-view',
    'jira-agile/rapid/ui/plan/sprint-view',
    'jira-agile/rapid/ui/plan/plan-controls'
], function(
    formatter,
    logger,
    DarkFeatures,
    Deferred,
    $,
    _,
    Util,
    Brace,
    InlineBootstrap,
    PlanWidgetsGroup,
    AgileIssueTypes,
    AgileDescription,
    DefaultTrigger,
    PlanController,
    BacklogController,
    BacklogModel,
    SprintBacklogView,
    SprintView,
    PlanControls
) {
    "use strict";

    var initialiser = {
        // The init function is defined so that consumers don't get
        // any '<thing>.init is not a function' JS errors if the feature
        // is disabled! (EEX-90)
        init: $.noop
    };

    if (DarkFeatures.isEnabled("jira.inline.issue.create.disable")){
        // Be sure to return the module definition! (EEX-90)
        return initialiser;
    }

    var TriggerView = DefaultTrigger.extend({
        serializeData: function serializeData() {
            return {
                cssClass: "iic-agile-plan"
            };
        }
    });

    var SectionModel = Brace.Model.extend({
        namedAttributes: {
            id: "string",
            translatedName: "string"
        }
    });
    var SectionCollection = Brace.Collection.extend({
        model: SectionModel
    });

    var widgetSections = new SectionCollection();

    var AgileInlineBootstrap = InlineBootstrap.extend({
        issueTypesCollectionType: AgileIssueTypes
    });

    var agileBootstrap = new AgileInlineBootstrap();

    var widgets = new PlanWidgetsGroup({
        TriggerView: TriggerView,
        inlineBootstrap : agileBootstrap,
        widgetOptions: {
            descriptionView: AgileDescription
        },
        createdCallback: function createdCallback(widget, id) {
            if (id !== "backlog") {
                widget.setOverrides({
                    "Sprint": id
                });
            }
            widget.descriptionView.currentSectionId = ""+id;
            widget.descriptionView.allSections = widgetSections;
            widget.listenTo(widget.descriptionView.allSections, "refreshed", function onSectionsRefreshed() {
                this.render();
            });

            widget.descriptionView.on("selected", function onElementSelected(element) {
                var $el = $(element);
                var id = $el.data("section-id");
                widgets.activateWidget(widgets.getWidget(id).widget);
            });
        }
    });

    function ensureActiveFilters() {
        var activeFilters = _.filter(PlanControls.quickfilters.getQuickFilters(), function selectOnlyActiveQuickFilters(e) {return e.isActive;});
        var activeFiltersJQLs = _.map(activeFilters, function (e) {return e.query;});
        widgets.setFilterJQLs(activeFiltersJQLs);
    }

    function refreshWidgetSections() {
        ensureActiveFilters();
        widgetSections.reset();
        agileBootstrap.fireWhenInitialized(function () {
            if (agileBootstrap.getHasCreateIssuePermission()) {
                if (agileBootstrap.getHasScheduleIssuePermission()) {
                    BacklogModel.getFutureSprintModels().forEach(function eachSprintModel(sprintModel) {
                        widgetSections.add(new SectionModel({id: "" + sprintModel.sprintData.id, translatedName: sprintModel.sprintData.name }));
                    });
                }
                widgetSections.add(new SectionModel({id: "backlog", translatedName: formatter.I18n.getText("iic.backlog.name") }));
            }
        });
        widgetSections.trigger("refreshed");
    }

    /**
     * Indicates when the agile board's configuration and initial data has been retrieved from the server.
     * This is 'guesstimated' based on when the {@link external:"GH.BacklogModel"#calculateIssueLists} function is called,
     * which sets the value of {@link external:"GH.BacklogModel.backlogModel2"}.
     * These functions are invoked via the following process:
     *
     * - {@link external:"GH.BacklogModel"#calculateIssueLists}
     * - {@link external:"GH.BacklogModel"#setData}
     * - {@link external:"GH.BacklogController"#processData}
     * - {@link external:"GH.BacklogController"#loadData}
     * - Various callers from here.
     */
    var dataLoaded = new Deferred();

    function shouldActivateBacklogWidget(widget) {
        // If the widget is in STATE_LOCK_PERM_INACTIVE state, we should not activate it
        // as it will open a create issue dialog automatically
        if (!widget || widget.model.getLockState() === widget.model.constants.STATE_LOCK_PERM_INACTIVE) {
            return false;
        }
        // Do not attempt to load when the backlog model's data is not actually populated.
        if (dataLoaded.state() === "pending") {
            return false;
        }
        var backlogIssueList = BacklogModel.getBacklogModel2().getIssueList();
        var issueListStats = GH.BacklogStatistics.getIssueListStats(backlogIssueList, false);
        return (issueListStats.totalCount === 0 && BacklogModel.getSprintModels().length === 0);
    }

    function renderBacklogWidget() {
        ensureActiveFilters();
        agileBootstrap.fireWhenInitialized(function () {
            dataLoaded.then(function initializeWidgetInBacklog() {
                if (agileBootstrap.getHasCreateIssuePermission()) {
                    var backlogWidget = widgets.getWidget("backlog");
                    backlogWidget.render($("#ghx-content-group .ghx-backlog-group .js-issue-list"));
                    if (shouldActivateBacklogWidget(backlogWidget.widget)) {
                        var model = backlogWidget.widget.model;
                        if (model.getIssueTypes()) {
                            model.setSelectedIssueType(model.getIssueTypes().at(0));
                        }
                        widgets.activateWidget(backlogWidget.widget);
                    }
                }
            });
        });

        logger.trace("inline.create.backlog.rendered");
    }

    var lastRapidViewId;
    function fetchBoardJql() {
        if(lastRapidViewId !== GH.RapidBoard.State.getRapidViewId()) {
            lastRapidViewId = GH.RapidBoard.State.getRapidViewId();
            logger.trace("inline.create.boardjql.fetching");
            GH.Ajax.get({
                url: '/rapidviewconfig/editmodel.json',
                data: {rapidViewId: GH.RapidBoard.State.getRapidViewId()},
                deferErrorHandling: true
            }, 'rapidBoardJQL').done(function onBoardJqlFetched(model) {
                var currentJQL = model.filterConfig.query;
                widgets.setBoardJQL(currentJQL);
                logger.trace("inline.create.boardjql.fetched");
            });
        }
    }
    function renderSprintWidget(result, sprintModel) {
        ensureActiveFilters();
        if (!sprintModel.isActiveSprint()) {
            var sprintId = sprintModel.getSprintData().id;
            agileBootstrap.fireWhenInitialized(function initializeWidgetInSprint() {
                if (agileBootstrap.getHasCreateIssuePermission() && agileBootstrap.getHasScheduleIssuePermission()) {
                    widgets.getWidget(sprintId).render(result.find(".ghx-issues"));
                }
            });
        }
    }

    var activateIntegration = _.once(function() {
        Util.after(BacklogModel, "setData", function afterDataSet() {
            dataLoaded.resolve();
            refreshWidgetSections();
        });
        Util.after(BacklogModel, "updateDataForSprint", refreshWidgetSections);
        Util.after(SprintView, "renderAllSprints", refreshWidgetSections);
        Util.after(SprintBacklogView, "renderBacklog", renderBacklogWidget);
        Util.after(PlanControls.quickfilters, "setActiveQuickFilters", ensureActiveFilters);

        Util.after(SprintBacklogView, "updateBacklog", renderBacklogWidget);
        Util.after(BacklogController, "updateEpicFiltering", renderBacklogWidget);
        Util.after(BacklogController, "updateVersionFiltering", renderBacklogWidget);
        Util.after(SprintView, "_renderSprintContent", renderSprintWidget);

        var sprintModels = BacklogModel.getSprintModels();

        fetchBoardJql();
        renderBacklogWidget();
        sprintModels.forEach(function(model) {
            var $sprintContainer = $('.js-sprint-container[data-sprint-id="' + model.getSprintId() + '"]');
            renderSprintWidget($sprintContainer, model);
        });
    });

    // Set the initialiser's real init method, so it's not just a no-op.
    initialiser.init = _.once(function() {
        if (PlanController.isActive()) {
            activateIntegration();
        }
        else {
            Util.after(PlanController, 'initializePlanMode', activateIntegration);
        }
    });

    return initialiser;
});
