#
#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.
# 
# Mercurial Extension, supporting extraction of the information that FishEye requires
'''felog

prints the information about the given commit that is required by FishEye'''

from mercurial import hg, commands, util, mdiff, cmdutil

try:
        hgversion = util.version()
except:
        from mercurial.version import version as v
        hgversion = v.get_version()

# via http://mercurial.selenic.com/wiki/ApiChanges
# Various functions have been moved from cmdutil.py to scmutil.py,
# including revrange/revsingle/revpair and match/matchall/matchfiles
if hgversion >= '1.9':
    from mercurial import scmutil
    utilmodule = scmutil
else:
    utilmodule = cmdutil

longFormats = {
    'commit':           'Commit Id:      %s\n',
    'rev' :             'Rev Id:         %s\n',
    'parent':           'Parents:        %s\n',
    'branch':           'Branch:         %s\n',
    'author':           'Author:         %s\n',
    'date':             'Date:           %s%s\n',
    'tags':             'Tags:           %s\n',
    'msg':              'Log Msg:        ',
    'file':             'File:           %s\n',
    'fileRev':          '  Rev:          %s\n',
    'fileStatus':       '  Status:       %s\n',
    'fileSize':         '  Size:         %s\n',
    'fileFlags':        '  Flags:        %s\n',
    'fileBinary':       '  Binary:       %s\n',
    'fileParent0':      '  Parent0:      %s %s\n',
    'fileParent1':      '  Parent1:      %s %s\n',
    'fileMergeFrom':    '  MergeParent:  %s\n',
    'fileMergeReplace': '  MergeReplace: %s\n',
    'diff':             '  Diff:         ',
}
shortFormats = {
    'commit':           'C:%s\n',
    'rev' :             'R:%s\n',
    'parent':           'P:%s\n',
    'branch':           'B:%s\n',
    'author':           'A:%s\n',
    'date':             'D:%s%s\n',
    'tags':             'T:%s\n',
    'msg':              'M:',
    'file':             'F:%s\n',
    'fileRev':          'FR:%s\n',
    'fileStatus':       'FT:%s\n',
    'fileSize':         'FS:%s\n',
    'fileFlags':        'FF:%s\n',
    'fileBinary':       'FB:%s\n',
    'fileParent0':      'FP0:%s %s\n',
    'fileParent1':      'FP1:%s %s\n',
    'fileMergeFrom':    'FM:%s\n',
    'fileMergeReplace': 'FE:%s\n',
    'diff':             'FD:',
}

def felog(ui, repo, *nodes, **opts):
    """Print extended log information about given REVS
REVS can be revsets style queries.
The -c option is for repositories converted from svn (or others, but particularly by hgconvert and hgsubversion).
These repos can create file revisions in mercurial for merge commits in the source repository that reference files that add files 
to the commit that were created in other commits - normal hg does this only on a merge and the file revisions are from one
of the parents, but hgsubversion/hgconvert produce this for non-merge commits (typically, they were merges in the parent VCS,
but the conversion didn't map a merge commit from the source VCS.
The -c flag says to treat these differently and mark them as merged files rather than additions/modifications. It should only
be used up to the last converted commit, as after that thewre are situations where normal hg produces simlar circumstances"""
    if nodes:
        for n in nodes:
            # Expand each node to a revlist
            for r in utilmodule.revrange(repo, [n]):
                felognode(ui, repo, repo[r], **opts)
    else:
        for r in repo:
            felognode(ui, repo, repo[r], **opts)

def felognode(ui, repo, ctx, **opts):
    """Print FishEye required information about a REV"""
    if ui.verbose:
        f = longFormats
    else:
        f = shortFormats
    commitState = commitstate(ctx)
    parents = ctx.parents()
    if len(ctx.description()) > 20000:
        description = ctx.description()[:20000] + '...'
    else:
        description = ctx.description()

    # Information about the commit itself
    ui.write(f['commit'] % ctx.hex())
    ui.write(f['rev'] % ctx.rev())
    ui.write(f['branch'] % ctx.branch())
    ui.write(f['author'] % ctx.user())
    ui.write(f['date'] % tuple([str(d) for d in ctx.date()]))
    ui.write(f['tags'] % " ".join(ctx.tags()))
    writePrefixed(ui, f['msg'], description)
    diffParent(ui, repo, ctx, commitState, f, opts.get('patch'), opts.get('converted'), opts.get('merges'), opts.get('changesets'))
    
ADDED="Added"
DELETED="Deleted"
MOVED="Renamed"
COPIED="Copied"
MODIFIED="Modified"
METACHANGE="MetaChange"
MERGED="Merged"

def diffParent(ui, repo, ctx, commitState, format, diffs, svnConversion, merges, changesets):
    ui.write(format['parent'] % " ".join([formatRev(ui, p) for p in ctx.parents()]).strip())
    if not changesets:
        modfiles = modifiedFiles(ctx)
        for f in modfiles:
            processPath(ui, repo, ctx, commitState, format, diffs, svnConversion, f)
        if merges and len(ctx.parents()) == 2:
            processMerge(ui, repo, ctx, commitState, format, modfiles)

def processPath(ui, repo, ctx, commitState, format, diffs, svnConversion, f):
    state = pathstate(repo, ctx, commitState, svnConversion, f)
    if state.discard():
        ui.debug("discarding change to file: ", state.path(), "\n")
        return
    # Output some file state
    ui.write(format['file'] % f)
    if state.state() == MERGED or ui.verbose:
        ui.write(format['fileRev'] % formatRev(ui, state.filerevctx()))
    ui.write(format['fileStatus'] % state.status())
    ui.write(format['fileSize'] % state.size())
    ui.write(format['fileFlags'] % state.flags())
    ui.write(format['fileBinary'] % state.binary())
    # We want the changectx where the parent filerevs were last changed
    def printParent(index, fkey):
        if len(state.parents()) > index:
            p = state.parents()[index]
            ui.write(format[fkey] % ( formatRev(ui, repo[p.linkrev()]), p.path()))
    printParent(0, 'fileParent0')
    printParent(1, 'fileParent1')
    if diffs and state != MERGED:
        state.diff(ui, format['diff'], repo, ctx)

class commitstate(object):
    """Cache information about our commit"""
    def __init__(self, ctx):
        self._ctx = ctx
        self._commonAncestor = None
        self._gotCommonAncestor = False

    def isMerge(self):
        return len(self._ctx.parents()) > 1

    def commonAncestor(self):
        if not self.isMerge():
            return None
        if not self._gotCommonAncestor:
            self._commonAncestor = self._ctx.p1().ancestor(self._ctx.p2())
            self._gotCommonAncestor = True
        return self._commonAncestor

    def isExplicitDelete(self, path):
        if path in self._ctx:
            return false;
        if not self.isMerge():
            return path in self._ctx.p1()
        inp1 = path in self._ctx.p1()
        inp2 = path in self._ctx.p2()
        # If in both, true, if in neither, false
        if inp1 == inp2:
            return inp1
        # So here, in 1 but not the other
        anc = self.commonAncestor()
        return not anc or not path in anc

    def isMergedDelete(self, path):
        return (not path in self._ctx
                and self.isMerge()
                and not path in self._ctx.p2()
                and path in self._ctx.p1()
                and self.commonAncestor()
                and path in self.commonAncestor())

class pathstate(object):
    """Extract state information about the path from the repo for the given context"""
    def __init__(self, repo, ctx, commitState, svnconversion, path):
        self._path = path
        self._state = MODIFIED
        self._discard = False
        # The file is only not in the context if it was deleted
        if path in ctx:
            filectx = ctx[path]
            self._size = filectx.size()
            self._flags = filectx.flags()
            # Get the commit it was last changed in (should be ctx), but isn't if its a meta-change
            self._filerevctx = repo[filectx.linkrev()]
            if self._filerevctx != ctx:
                # Due to a bug in Mercurial, the filerev may not exist in the filerevctx: http://mercurial.selenic.com/bts/issue2699
                # In this case it will end up as MODIFIED
                # Otherwise:
                # It's a MetaChange (perms) if it's last mod wasn't this ctx and the flags have changed
                if path in self._filerevctx and filectx.flags() != self._filerevctx[path].flags():
                    self._state = METACHANGE
                # If the last mod is not this ctx (and it's not a PermChange), it's likely these are merged revisions
                # produced by hgsubversion and hgconvert, in which case we should maybe just ignore them or report them.
                #
                # It's possible these results are because it's the same diff and same parent rev, therefore it reuses
                # an already existing rev with the same hash. Maybe a file modded twice on a branch and merged (therefore
                # different diffs and parents) will result in a real new revision in the merge commit. - Could be just
                # natural behaviour of mercurial rather than something hgconvert/hgsubversion is doing.
                # Because it's not a filerev created in the dag of the parent, it gets added to changectx.files()
                elif svnconversion and len(ctx.parents()) < 2:
                    self._state = MERGED
            # We need to make sure here that the file parents order corresponds to the commits parents order
            self._parents = filectx.parents()
            if len(self._parents) > 1:
                fp1 = filectx.parents()[0]
                cp1 = ctx.p1()
                if fp1.path() in cp1 and fp1.linkrev() == cp1[fp1.path()].linkrev():
                    # Order is fine, leave it alone
                    self._parents = filectx.parents()
                else:
                    self._parents.reverse()
        else:
            self._size = 0
            self._flags = ''
            # Either a merged in delete, or deleted during the merge, treat both as deleted in this rev.
            self._filerevctx = ctx
            def getDeletedFileParents(ctx, file):
                parents = []
                for p in ctx.parents():
                    if file in p:
                        parents.append(p[file])
                if len(parents) == 2:
                    # Check if they both refer to same rev or one is the ancestor of another. check the linkrev or filerev, since
                    # the filectx itself could refer to the same but be different instances and i'm not sure about pythons ==
                    if parents[0].linkrev() == parents[1].linkrev():
                        # Both parents have the same filerev, just use parent0
                        parents.pop()
                return parents
            self._parents = getDeletedFileParents(ctx, path)
            self._state = DELETED
        self._fromPath = path
        # If it has no parents, it could have had a perm change in this rev, which results in a filectx with no parents,
        # so check if it was in a parent changectx
        if self._state != MERGED and len(self._parents) == 0 and not inParents(path, ctx):
            self._state = ADDED
        elif self._state != DELETED and self._state != MERGED:
            if len(self._parents) > 0 and self._parents[0].path() != path:
                self._fromPath = self._parents[0].path()
                if not self._fromPath in ctx:
                    self._state = MOVED
                else:
                    self._state = COPIED
        # Cope with when the change is a move, yet the parent path doesn't exist in the parent CS: seen this happen
        if self._state == MOVED and self._filerevctx != ctx and len(ctx.parents()) > 0 and not self._fromPath in ctx.p1():
            self._discard = True
        # Determine the binary state: cache the file content for the diff
        self._binary = False
        if self._state == DELETED:
            # If it's a merged delete, get rid of it
            if commitState.isMergedDelete(self._path):
                self._discard = True
            self._binary = util.binary(self._parents[0].data())
        else:
            self._content = ctx[self._path].data()
            self._binary = util.binary(self._content)
    
    def path(self):
        return self._path

    def state(self):
        return self._state

    def frompath(self):
        return self._fromPath

    def size(self):
        return self._size

    def flags(self):
        return self._flags

    def binary(self):
        return self._binary

    def parents(self):
        return self._parents

    def discard(self):
        return self._discard

    def filerevctx(self):
        """The changectx this path was last modified in"""
        return self._filerevctx

    def status(self):
        """The state as a string"""
        if self._state == MOVED:
            return "Move from %s" % self._fromPath
        elif self._state == COPIED:
            return "Copy from %s" % self._fromPath
        else:
            return self._state

    def diffParent(self, repo, ctx):
        """Pick a parent to diff against
        if path exists in a parent, diff against that (could be a perm change on an added file), other wise parents()[0]"""
        numcctxparents = len(ctx.parents())
        if numcctxparents == 0:
            # First commit - compare to 00000000000000
            return repo[0].p1()
        elif numcctxparents == 1:
            # compare to parent commit
            return ctx.p1()
        else:
            numfctxparents = len(self._parents)
            # Merge commit: Find a parent
            if self._state == METACHANGE:
                # Compare against the parent changectx which has the same entry in its manifest
                mfe = ctx.manifest()[self._path]
                for p in ctx.parents():
                    if self._path in p and p.manifest()[self._path] == mfe:
                        return p
                # Can we get here? Back off safely:
                return ctx.p1()
            elif numfctxparents == 2:
                # If file has 2 parents, compare to ctx.p1()
                return ctx.p1()
            elif numfctxparents == 1:
                # compare against the commit it last modded in
                return repo[self._parents[0].linkrev()]
            else:
                # no parents - add - compare to p1
                return ctx.p1()

    def diff(self, ui, prefix, repo, ctx):
        """Do a diff against the parent rev"""
        fromRev = self.diffParent(repo, ctx)
        extra = None
        def modeStr(fctx, pre):
            s = "644"
            if fctx.flags() == 'x':
                s = "755"
            return pre + " 100" + s
        def diffstr(path1, path2):
            # Bizarrely, if one side of the diff does not exist, hg diff --git prints that it is diffing the file to itself but then the +++/--- lines use /dev/null
            p1 = "a/" + path1
            p2 = "b/" + path2
            if path1 == "":
                p1 = "a/" + path2
            if path2 == "":
                p2 = "b/" + path1
            return "diff --git %s %s\n" % (p1, p2)
        # We try and avoid weird cases by not returning a diff if we can't find either version
        if self._state == ADDED:
            aFileName = ""
            aFileData = None
            aFileDate = None
            aBinary = False
            bFileName = self._path
            if not bFileName in ctx:
                ui.debug("Can't find ", bFileName, " in commit (", ctx.rev(), ") manifest, not diffing")
                return
            bFileCtx = ctx[bFileName]
            bFileData = self._content
            bFileDate = util.datestr(bFileCtx.date())
            bBinary = self._binary
            extra = modeStr(bFileCtx, "new file mode")
        elif self._state == DELETED:
            aFileName = self._path
            if not aFileName in fromRev:
                ui.debug("Can't find ", aFileName, " in parent commit (", fromRev.rev(), ") manifest, not diffing")
                return
            aFileCtx = fromRev[aFileName]
            aFileData = aFileCtx.data()
            aFileDate = util.datestr(aFileCtx.date())
            aBinary = util.binary(aFileData)
            bFileName = ""
            bFileData = None
            bFileDate = None
            bBinary = False
            extra = modeStr(aFileCtx, "deleted file mode")
        else:
            aFileName = self._fromPath
            if not aFileName in fromRev:
                ui.debug("Can't find ", aFileName, " in parent commit (", fromRev.rev(), ") manifest, not diffing")
                return
            aFileCtx = fromRev[aFileName]
            aFileData = aFileCtx.data()
            aFileDate = util.datestr(aFileCtx.date())
            aBinary = util.binary(aFileData)
            bFileName = self._path
            if not bFileName in ctx:
                ui.debug("Can't find ", bFileName, " in commit (", ctx.rev(), ") manifest, not diffing")
                return
            bFileCtx = ctx[bFileName]
            bFileData = self._content
            bFileDate = util.datestr(bFileCtx.date())
            bBinary = self._binary
        ui.write(prefix, diffstr(aFileName, bFileName))
        if self._state == MOVED or self._state == COPIED:
            msg = "rename"
            if self._state == COPIED:
                msg = "copy"
            ui.write(prefix, msg, " from ", aFileName, "\n")
            ui.write(prefix, msg, " to ", bFileName, "\n")
        if extra:
            ui.write(prefix, extra, '\n')
        diffopts = mdiff.diffopts()
        diffopts.context = 0
        diffopts.nodates = True
        diffopts.git = True
        if aBinary and not bBinary and bFileData:
            # Make it look like we added all of b, cause it went binary -> ascii
            aFileData = ""
            aBinary = False
        if aFileData == bFileData:
            # do nothing
            pass
        elif aBinary or bBinary:
            ui.write(prefix, "Binary file ", self._path, " has changed\n")
        else:
            if hgversion >= '2.5':
                writePrefixed(ui, prefix, mdiff.unidiff(aFileData, aFileDate, bFileData, bFileDate, aFileName, bFileName, diffopts))
            else:
                writePrefixed(ui, prefix, mdiff.unidiff(aFileData, aFileDate, bFileData, bFileDate, aFileName, bFileName, None, diffopts))
        return


def writePrefixed(ui, prefix, output):
    """Write the output but prefix each line with prefix. Lines are terminated on windows, mac or unix newlines"""
    skipLinefeed = False
    writePrefix = True
    for c in output:
        if writePrefix:
            ui.write(prefix)
            writePrefix = False
        if c == '\n':
            if not skipLinefeed:
                writePrefix = True
                ui.write('\n')
        elif c == '\r':
            skipLinefeed = True
            writePrefix = True
            ui.write('\n')
        else:
            if skipLinefeed:
                skipLinefeed = False
            ui.write(c)
    if not writePrefix:
        # Write a terminating newline if there isn't one
        ui.write('\n')

def modifiedFiles(ctx):
    """Detect renames and remove the renamed file from the file list"""
    modfiles = ctx.files()[:]
    for f in ctx.files():
        if f in ctx and len(ctx[f].parents()) > 0:
            fromPath = ctx[f].parents()[0].path()
            if fromPath != f and fromPath in modfiles and not fromPath in ctx:
                modfiles.remove(fromPath)
    return modfiles

def inParents(f, ctx):
    for p in ctx.parents():
        if f in p:
            return True
    return False

def formatRev(ui, ctx):
    if not ctx:
        return ""
    if ui.verbose:
        return "%s:%s" % (ctx.rev(), ctx.hex())
    return ctx.hex()

def processMerge(ui, repo, ctx, commitState, format, modfiles):
    """Produce a list of files merged in from parents"""
    p1 = ctx.p1()
    p2 = ctx.p2()
    def same(path, c1, c2):
        return path in c1 and path in c2 and c1[path] == c2[path]
    def versionIn(path, mf):
        if path in mf:
            return repo[mf[path]]
    fromp1 = []
    fromp2 = []
    # It's twice as fast to grab the whole manifests than it is to iterate one file at a time checking it against both parents (FECRU-539)
    mergemf = {}
    for path in ctx:
        mergemf[path] = ctx[path].linkrev()
    p1mf = {}
    for path in p1:
        p1mf[path] = p1[path].linkrev()
    p2mf = {}
    for path in p2:
        p2mf[path] = p2[path].linkrev()
    for path in mergemf:
        # detect files added/merged from parents
        if not path in modfiles:
            samep1 = same(path, mergemf, p1mf)
            samep2 = same(path, mergemf, p2mf)
            if samep1 and not samep2:
                fromp1.append((path, versionIn(path, p2mf)))
            elif not samep1 and samep2:
                fromp2.append((path, versionIn(path, p1mf)))
    def printModifiedPath(path, replace, ctx):
        ui.write(format['file'] % path)
        ui.write(format['fileRev'] % formatRev(ui, repo[ctx[path].linkrev()]))
        ui.write(format['fileStatus'] % MERGED)
        ui.write(format['fileMergeFrom'] % formatRev(ui, ctx))
        ui.write(format['fileMergeReplace'] % formatRev(ui, replace))
    # Sort the merged files by alphabetical order
    fromp1.sort(key=lambda p: p[0])
    fromp2.sort(key=lambda p: p[0])
    for pathInfo in fromp1:
        printModifiedPath(pathInfo[0], pathInfo[1], ctx.p1())
    for pathInfo in fromp2:
        printModifiedPath(pathInfo[0], pathInfo[1], ctx.p2())
#    from collections import deque
#    # Look for deletions:
#    deletedonp1 = []
#    deletedonp2 = []
#    # files() always has the deletions being merged in (from p2)
#    for f in ctx.files():
#        if commitState.isMergedDelete(f):
#            # I think they will all be deleted on p2, but be paranoid
#            if not f in p2mf:
#                deletedonp2.append((f, versionIn(f, p1mf)))
#            else:
#                deletedonp1.append((f, versionIn(f, p2mf)))
#    # Files deleted on parent2 are reported in files, but those deleted in parent1 are not (FE-3010)
#    for path in p2mf:
#        if not path in mergemf and not path in p1mf:
#            deletedonp1.append((path, versionIn(path, p2mf)))
#    # Really need to consider if this is worth it - the danger is of swelling the process memory
#    # It might just be better to let FishEye try to figure out what the deleted revision is, or
#    # Just note it was deleted
#    commitStateCache = {}
#    def findDelete(path, ctx):
#        """Look for where the path was deleted in the ancestry tree"""
#        toCheck=deque([ctx.rev()])
#        while toCheck:
#            c = repo[toCheck.popleft()]
#            #print "checking ", c.rev()
#            # Adding this massively reduces the memory footprint: without it, this is unusable
#            if path in c.files():
#                commitState = commitStateatecache.get(c.rev())
#                if not commitState:
#                    commitState = commitstate(c)
#                    commitStateCache[c.rev()] = commitState
#                if commitState.isExplicitDelete(path):
#                    return c
#            for p in c.parents():
#                if not path in p and not p.rev() in toCheck:
#                    toCheck.append(p.rev())
#                    #print "tocheck: ", toCheck
#        return None
#    def printDeletedPath(path, replace, ctx):
#        ui.write(format['file'] % path)
#        ui.write(format['fileRev'] % formatRev(ui, findDelete(path, ctx)))
#        ui.write(format['fileStatus'] % MERGED)
#        ui.write(format['fileMergeFrom'] % formatRev(ui, ctx))
#        ui.write(format['fileMergeReplace'] % formatRev(ui, replace))
#    for pathInfo in deletedonp1:
#        printDeletedPath(pathInfo[0], pathInfo[1], ctx.p1())
#    for pathInfo in deletedonp2:
#        printDeletedPath(pathInfo[0], pathInfo[1], ctx.p2())

def feparents(ui, repo, node, **opts):
    """For a REV, list the last modified rev of all the present files
Can be a tag, branch, tip or explicit revision"""
    ctx = repo[node]
    revstrs = {}
    for file in ctx:
        rev = ctx[file].linkrev()
        if not rev in revstrs:
            revstrs[rev] = formatRev(ui, repo[rev])
        ui.write(revstrs[rev], " ", file, "\n")

def fecheck(ui, repo, *nodes, **opts):
    """For a set of revisions, run some checks"""
    if nodes:
        for n in nodes:
            # Expand each node to a revlist
            for r in utilmodule.revrange(repo, [n]):
                fechecknode(ui, repo, repo[r], **opts)
    else:
        for r in repo:
            fechecknode(ui, repo, repo[r], **opts)

def fechecknode(ui, repo, ctx, **opts):
    ui.debug("Checking ", ctx.rev(), "\n")
    if opts.get('dupmods'):
        for f in modifiedFiles(ctx):
            if f in ctx and ctx[f].linkrev() != ctx.rev():
                ui.write("%s; %s linkrev=%s\n" % (formatRev(ui, ctx), f, formatRev(ui, repo[ctx[f].linkrev()])))
    if opts.get('delfile2parents'):
        for f in modifiedFiles(ctx):
            # Merge and delete and exists in both parents
            if len(ctx.parents()) == 2 and not f in ctx and f in ctx.parents()[0] and f in ctx.parents()[1]:
                ui.write("%s; %s: %s\n" % (formatRev(ui, ctx), f, ctx.description()))
    if opts.get('delmerges'):
        if len(ctx.parents()) == 2:
            # With this 422.917s
            # Without: 72.167s
            ancestor = ctx.p1().ancestor(ctx.p2())
            for f in modifiedFiles(ctx):
                if not f in ctx and f in ancestor:
                    if f in ctx.parents()[0] and not f in ctx.parents()[1]:
                        ui.write(formatRev(ui, ctx), "; ", f, " deleted from p1\n")
                    elif f in ctx.parents()[1] and not f in ctx.parents()[0]:
                        ui.write(formatRev(ui, ctx), "; ", f, " deleted from p0\n")

cmdtable = {
    "felog":
        (felog,
         [('p', 'patch', None, "show patch"),
          ('c', 'converted', None, "look for artifacts of a hgconvert/hgsubversion repository"),
          ('m', 'merges', None, "produce merge information"),
          ('s', 'changesets', None, "only log changesets, not the paths in the commit")],
         "[-p][-c][-m][-s] REV"),
    "feparents":	(feparents, [], "REV"),
    "fecheck":		(fecheck, [
            ('d', 'dupmods', None, "Check for mods not first made in current commit"),
            ('p', 'delfile2parents', None, "Chack for deleted files with 2 parents"),
            ('m', 'delmerges', None , "Check for files deleted in merges")
            ], "REVSET")
}
