FECRU.UTIL.dateFormatter = (function (moment) {
    'use strict';

    /*
     Ideally this should follow https://design.atlassian.com/2.1/product/patterns/date-formats/ and reuse
     AUI helpers, if available.
     For now following same date formatting rules as used by Crucible core, see
     com.cenqua.fisheye.web.tags.DateTag#FORMAT_FULL
     */

    var checkDateType = function (date) {
        if (!moment.isMoment(date)) {
            throw new Error('Target date must be an instance of moment.js date');
        }
        return date;
    };

    var full = function (date) {
        return checkDateType(date).format('DD MMMM YYYY H:mm');
    };

    var iso = function (date) {
        return checkDateType(date).toISOString();
    };

    var brief = function (date, currentDate) {
        if (typeof currentDate === 'undefined') {
            currentDate = moment();
        } else if (!moment.isMoment(currentDate)) {
            currentDate = moment(currentDate);
        }
        checkDateType(date);

        var format = 'DD MMM YY';
        if (date.isSame(currentDate, 'day')) {
            format = 'HH:mm';
        } else if (date.isSame(currentDate, 'year')) {
            format = 'DD MMM';
        }
        return date.format(format);
    };

    var relative = (function () {
        // relative date formatter
        // based on spec https://design.atlassian.com/2.0/product/patterns/date-formats/
        var diffInMinutes = function (diff) {
            return Math.floor(diff / 60);
        };

        var diffInHours = function (diff) {
            return Math.floor(diff / 3600);
        };

        var diffInDays = function (diff) {
            return Math.floor(diff / (3600 * 24));
        };

        var TEST_FORMAT = 'YYYY-MM-DD';
        var isYesterday = function (date, currentDate) {
            if (date.tz()) {
                currentDate = currentDate.tz(date.tz());
            }
            return date.format(TEST_FORMAT) === currentDate.subtract(1, 'days').format(TEST_FORMAT);
        };

        return function (date, currentDate) {
            if (!moment.isMoment(date)) {
                throw new Error('Target date must be an instance of moment.js date');
            }

            if (typeof currentDate === 'undefined') {
                currentDate = moment();
            } else if (!moment.isMoment(currentDate)) {
                currentDate = moment(currentDate);
            }

            var dateTimestamp = date.unix();
            var diff = currentDate.unix() - dateTimestamp;

            var minutes = diffInMinutes(diff);
            var hours = diffInHours(diff);
            var days = diffInDays(diff);

            switch (true) {
                case minutes === 0:
                    return 'just now';

                case minutes === 1:
                    return '1 minute ago';

                case hours === 0:
                    return minutes + ' minutes ago';

                case hours === 1:
                    return '1 hour ago';

                case isYesterday(date, currentDate):
                    return 'yesterday';

                case days === 0:
                    return hours + ' hours ago';

                case days === 1:
                    return '2 days ago';

                case days < 7:
                    return days + ' days ago';

                case days === 7:
                    return '1 week ago';
            }

            return date.format('D MMM YYYY');
        };
    })();

    /**
     * @param {moment} date
     * @returns {string}
     */
    var formatter = function (date) {
        return full(date);
    };
    /**
     * @param {moment} date
     * @returns {string}
     */
    formatter.full = full;
    /**
     * @param {moment} date
     * @returns {string}
     */
    formatter.iso = iso;
    /**
     * @param {moment} date
     * @param {(moment|Date)} [currentDate]
     * @returns {string}
     */
    formatter.brief = brief;
    /**
     * @param {moment} date
     * @param {(moment|Date)} [currentDate]
     * @returns {string}
     */
    formatter.relative = relative;

    return formatter;
})(moment);