/*
 * Viewport - jQuery selectors for finding elements in viewport
 *
 * Copyright (c) 2008-2009 Mika Tuupola
 *
 * Licensed under the MIT license:
 *   http://www.opensource.org/licenses/mit-license.php
 *
 * Project home:
 *  http://www.appelsiini.net/projects/viewport
 *
 * Modifications:
 * - monitor the #frx-pane instead of the window
 * - threshold is an optional param, e.g., ':in-viewport(100)'
 * 
 * Optimizations:
 * - avoid redundant lookups
 * - avoid horizontal calculation (via ':in-viewport-vert')
 */
(function($) {

    //map of container id to {$container, containerOffset, containerScrollTop, containerScrollLeft}
    var containers = {};
    var windowContainerSet; //have to keep this separate because we dont have an id to match against it
    var $element;
    var elementOffset;

    function init(element, containerId) {
        var containerSet;
        if (containerId) {
            containerSet = containers[containerId];
        } else {
            containerSet = windowContainerSet;
        }
        var $container;
        var containerOffset;
        if (!containerSet) {
            containerSet = {};
            $container = $(containerId ? '#' + containerId : window);
            containerSet.container = $container;
        } else {
            $container = containerSet.container;
        }
        containerSet.containerOffset = containerOffset = $container.offset();
        var containerScrollTop = containerSet.containerScrollTop = $container.scrollTop();
        var containerScrollLeft = containerSet.containerScrollLeft = $container.scrollLeft();

        $element = $(element);

        var rawOffset = $element.offset();
        elementOffset = {
            top: rawOffset.top - (containerOffset ? containerOffset.top : 0) + containerScrollTop,
            left: rawOffset.left - (containerOffset ? containerOffset.left : 0) + containerScrollLeft
        };

        if (containerId) {
            containers[containerId] = containerSet;
        } else {
            windowContainerSet = containerSet;
        }
        return containerSet;
    }

    function isBelowTheFold(settings, containerSet) {
        var fold = containerSet.container.height() + containerSet.containerScrollTop;
        return fold - settings.threshold <= elementOffset.top;
    }

    $.belowthefold = function(element, settings, containerId) {
        var containerSet = init(element, containerId);
        return isBelowTheFold(settings, containerSet);
    };

    function isAboveTheTop(settings, containerSet) {
        var top = containerSet.containerScrollTop;
        return top + settings.threshold >= elementOffset.top + $element.height();
    };

    $.abovethetop = function(element, settings, containerId) {
        var containerSet = init(element, containerId);
        return isAboveTheTop(settings, containerSet);
    };

    function isRightOfScreen(settings, containerSet) {
        var fold = containerSet.container.width() + containerSet.containerScrollLeft;
        return fold - settings.threshold <= elementOffset.left;
    }

    $.rightofscreen = function(element, settings, containerId) {
        var containerSet = init(element, containerId);
        return isRightOfScreen(settings, containerSet);
    };

    function isLeftOfScreen(settings, containerSet) {
        var left = containerSet.containerScrollLeft;
        return left + settings.threshold >= elementOffset.left + $element.width();
    }

    $.leftofscreen = function(element, settings, containerId) {
        var containerSet = init(element, containerId);
        return isLeftOfScreen(settings, containerSet);
    };

    $.inviewport = function(element, settings, containerId) {
        var containerSet = init(element, containerId);
        return !isLeftOfScreen(settings, containerSet) && !isRightOfScreen(settings, containerSet) && !isBelowTheFold(settings, containerSet) && !isAboveTheTop(settings, containerSet);
    };

    $.inviewportvert = function(element, settings, containerId) {
        var containerSet = init(element, containerId);
        return !isBelowTheFold(settings, containerSet) && !isAboveTheTop(settings, containerSet);
    };

    $.extend($.expr[':'], {
        "below-the-fold": function(a, i, m) {
            return $.belowthefold(a, {threshold : parseInt(m[3], 10) || 0}, m[3].split(' ')[1]);
        },
        "above-the-top": function(a, i, m) {
            return $.abovethetop(a, {threshold : parseInt(m[3], 10) || 0}, m[3].split(' ')[1]);
        },
        "left-of-screen": function(a, i, m) {
            return $.leftofscreen(a, {threshold : parseInt(m[3], 10) || 0}, m[3].split(' ')[1]);
        },
        "right-of-screen": function(a, i, m) {
            return $.rightofscreen(a, {threshold : parseInt(m[3], 10) || 0}, m[3].split(' ')[1]);
        },
        "in-viewport": function(a, i, m) {
            return $.inviewport(a, {threshold : parseInt(m[3], 10) || 0}, m[3].split(' ')[1]);
        },
        "in-viewport-vert": function(a, i, m) {
            return $.inviewportvert(a, {threshold : parseInt(m[3], 10) || 0}, m[3].split(' ')[1]);
        }
    });

})(jQuery);
/*[{!jquery_viewport_js_hvek55t!}]*/