/**
 *
 * Will create an inline editable select field. Click on the icon next to the field
 * to view the select and change the value.
 *
 * Use on a select:
 *
 *      <select>
 *          //options
 *      </select>
 *
 * to produce a structure like:
 *
 * <span class="inline-select">
 *      <span class="field">Selected Item Text</span><a>Edit icon</a>
 *      <select>
 *          //options
 *      </select>
 *      <a>stop editing icon</a>
 * </span>
 *
 * usage: var x = $(selector).inlineSelect(options);
 *
 * Custom bound events:
 *
 *  usage: x.trigger("eventName");
 *
 * inlineSelect.setValue - sets the value in the field to the selected value
 * inlineSelect.showEditMode - shows the select and button to close it
 * inlineSelect.showNormalMode - shows the field and the button to edit it
 * inlineSelect.showSpinner - shows a spinner in place of the close button
 * inlineSelect.hideSpinner - hides the spinner in place of the close button
 *
 * Options:
 *
 *  onChange: function(selected) {
 *            }
 *
 *            function to be called when the value of the select is changed.
 *
 *            otherwise, the function will pass in the selected option:
 *
 *            function() {
 *               inlineSelect.setValue();
 *               inlineSelect.showEdit();
 *            }
 *
 */
;
(function($) {

    $.fn.extend({
        inlineSelect: function (opts) {
            var $select = this;

            var $container = $("<span class='inline-select' />");
            $select.wrap($container);
            var $label = $("<span class='field'>" + $select.find("option:selected").text() + "</span>");
            $select.before($label);
            
            var $edit = $("<a class='change button' title='Click to change.'>Change</a>")
                    .insertAfter($label);
            var $cancel = $("<a class='cancel button' title='Click to cancel'>Cancel</a>")
                    .insertAfter($select);

            function showEditMode() {
                $select.show();
                $cancel.show();
                $label.hide();
                $edit.hide();
            }

            function showNormalMode() {
                $select.hide();
                $cancel.hide();
                $edit.show();
                $label.show();
            }

            function showSpinner() {
                $cancel.addClass("spinner");
            }

            function setValue() {
                var $selected = $select.children("option:selected");
                $label.html($selected.text() || $selected.val());
            }

            function hideSpinner() {
                $cancel.removeClass("spinner");
            }

            $container.bind("inlineSelect.setValue", setValue);
            $container.bind("inlineSelect.showEditMode", showEditMode);
            $container.bind("inlineSelect.showNormalMode", showNormalMode);
            $container.bind("inlineSelect.showSpinner", showSpinner);
            $container.bind("inlineSelect.hideSpinner", hideSpinner);

            $edit.click(function(e) {
                showEditMode();

            });

            $cancel.click(function (e) {
                showNormalMode();
            });


            showNormalMode();


            if (opts && opts.onChange) {
                $container.onChange = opts.onChange;
            } else {
                $container.onChange = function() {
                    setValue();
                    showNormalMode();
                }
            }

            $select.change(function() {
                $container.onChange($select.children("option:selected"));
            });


            return $container;
        }
    });
})(jQuery);
/*[{!jquery_fecru_inlineSelect_js_exqs55h!}]*/