/**
 *  Abbreviates a path by removing path segments from the center outwards
 *
 * Options = {
 *      boundingWidthGetter : function($path) { * return the maximum width for this element * }
 * ,    growingElementGetter : function($path) { * return the current width being measured * }
 * ,    separatorText : string that separates path components (default is "/")
 * ,    replacement :  string that is used in place of omitted sections (default is "...")
 * ,    widthCacheKey : abbreviatePath caches previous lengths.  if something has changed within the bounding element, modify the widthCacheKey to match the new state
 * ,    directionHint : if you know for a fact that the bounds are getting smaller/larger, use 'smaller'/'larger' as a direction hint
 * ,    resetBoundsWidth : for performance, abbreviatePath will only reread the bounds if resetBoundsWidth is true.
 * }
 */

(function($) {
    // IE reflows cause resizes in weird cases.
    // this var is used to ignore those phantom resizes.
    // see http://bugs.jquery.com/ticket/4097
    var disableAll = false;

    var bufferSpace = 10;

    function hideNext(components, numberToCut) {
        if (!numberToCut) { //hide the replacement
            components.filter('.replacement').addClass('hidden');
        } else {
            var replacementIndex = Math.floor((components.length - 1) / 2) + 1;
            if (numberToCut === 1) { // show replacement
                components.eq(replacementIndex).removeClass('hidden');
            }
            if (numberToCut % 2) { // cut off the front half
                components.eq(replacementIndex - Math.ceil(numberToCut / 2)).addClass('hidden');
            } else { // cut off the back half
                components.eq(replacementIndex + (numberToCut / 2)).addClass('hidden');
            }
        }
    }

    function checkIfFits(widthCache, numberCut, boundingWidth, grower) {
        if (!widthCache.hasOwnProperty(numberCut)) {
            widthCache[numberCut] = grower.width();
        }
        return boundingWidth >= (widthCache[numberCut] + bufferSpace);
    }

    function shortenPaths(hintDirection, $paths, options) {
        if (!$paths.length) {
            return;
        }
        if (!boundsWidth) {
            $paths.addClass('hidden');
            var $path = $paths.first(),
                pathData = $path.data('abbreviatePath');
            boundsWidth = pathData.boundingWidthGetter($path);
            $paths.removeClass('hidden');
        }
        for (var i = 0, l = $paths.size(); i < l; i++) {
            var $path = $paths.eq(i),
                pathData = $path.data('abbreviatePath'),
                $growEl = pathData.grower,
                components = pathData.components,
                lastCut = pathData.lastCut,
                widthCache = pathData.widthCache;

            var componentLength = components.length - 1, // replacement doesn't count
                numberToCut = 0;

            if (hintDirection && hintDirection === 'smaller') { //start where we left off
                numberToCut = lastCut;
                while (numberToCut + 2 < componentLength) {
                    hideNext(components, numberToCut);
                    pathData.lastCut = numberToCut;
                    if (checkIfFits(widthCache, numberToCut, boundsWidth, $growEl)) {
                        break;
                    }
                    numberToCut++;
                }
            } else {
                lastCut && components.removeClass('hidden'); //if we've shortened it, reset
                do { // always do it at least once so we can hide the replacement if unneeded
                    hideNext(components, numberToCut);
                    pathData.lastCut = numberToCut;
                    if (checkIfFits(widthCache, numberToCut, boundsWidth, $growEl)) {
                        break;
                    }
                    numberToCut++;
                } while (numberToCut + 2 < componentLength);
            }
        }
    }

    function getPathComponents($pathElement, separatorText, $replacement) {
        var pathChildren = $pathElement.children();
        $('<span class="replacement" />').
                append($replacement).
                addClass('hidden').
                insertAfter(pathChildren.eq(Math.floor(pathChildren.length / 2)));
        return $pathElement.children();
    }

    var boundsWidth;
    var cacheGetter = function($this, dataObj, dataKey, costlyGet) {
        var value = dataObj[dataKey];
        if (!value) {
            dataObj[dataKey] = value = costlyGet($this);
        }
        return value;
    },
    growingElementGetter = function($this, dataObj, costlyGet) {
        return cacheGetter($this, dataObj, 'grower-abbreviatePath', costlyGet);
    },
    widthCacheGetter = function($this, dataObj, which) {
        return cacheGetter($this, dataObj, 'widthCache-' + which + '-abbreviatePath', function() { return {}; });
    };

    function initPaths($paths, options, $replacement) {

        for (var i = 0, l = $paths.size(); i < l; i++) {
            var $path = $paths.eq(i),
                pathData = $path.data('abbreviatePath'),
                hadData = !!pathData;
            pathData = pathData || {};

            var $growEl = growingElementGetter($path, pathData, options.growingElementGetter),
                components = pathData.components || getPathComponents($path, options.separatorText, $replacement.clone());

            if (options.resetBoundsWidth) {
                boundsWidth = null;
            }

            pathData.boundingWidthGetter = options.boundingWidthGetter;
            pathData.grower = $growEl;
            pathData.components = components;
            pathData.disabled = components.length < 3;
            pathData.widthCache = widthCacheGetter($path, pathData, options.widthCacheKey);
            pathData.separatorText = options.separatorText;
            pathData.lastCut = pathData.lastCut || 0;
            pathData.componentLengthCache = pathData.componentLengthCache || [];

            if (!hadData) {
                $path.data('abbreviatePath', pathData);
            }
        }
    }

    function isString(object) {
        return typeof(object) === 'string' || object instanceof String;
    }

    $.fn.abbreviatePath = function(optsOrKey) {
        if (!this.length || disableAll) {
            return this;
        }
        disableAll = true;

        var options = $.extend({}, $.fn.abbreviatePath.defaults, optsOrKey),
            $replacement = isString(options.replacement) ?
                                $('<span />').text(options.replacement) :
                                $(options.replacement);

        initPaths(this, options, $replacement);
        shortenPaths(options.directionHint, this.filter(function() {
            return !$(this).data('abbreviatePath').disabled;
        }), options);

        disableAll = false;

        return this;
    };
    $.fn.abbreviatePath.defaults = {
        boundingWidthGetter : function($path) {
            return $path.closest('.abbreviatePath-parent').width();
        }
    ,   growingElementGetter : function($path) {
            return $path;
        }
    ,   separatorText : '/'
    ,   replacement : '...'
    ,   widthCacheKey : 0
    };

})(window.AJS ? AJS.$ : jQuery);/*[{!jquery_fecru_abbreviatePath_js_lvim55e!}]*/